package de.pheasn.blockown.database;

import de.pheasn.blockown.Output;
import de.pheasn.blockown.Ownable;
import de.pheasn.blockown.OwnedBlock;
import de.pheasn.blockown.OwnedEntity;
import de.pheasn.blockown.User;

import java.io.File;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.UUID;

public class SqliteDatabase extends CachedDatabase {

	private static final String FILENAME = "world.db";
	private final File pluginDataFolder;

	public SqliteDatabase(Output output, File pluginFolder) throws SQLException, ClassNotFoundException {
		super(output, pluginFolder);
		this.pluginDataFolder = pluginFolder.getAbsoluteFile();
		Class.forName("org.sqlite.JDBC");
		connection = connect();
		createTables();
	}

	private Connection connect() throws SQLException {
		File file = new File(pluginDataFolder, FILENAME);
		String url = "jdbc:sqlite:" + file.getPath();
		return DriverManager.getConnection(url);
	}

	@Override
	String[] generateCreateTableQueries() {
		String[] queries = new String[2];
		queries[0] = generateCreateBlockTableQuery();
		queries[1] = generateCreateEntityTableQuery();
		return queries;
	}

	private String generateCreateBlockTableQuery() {
		return "CREATE TABLE IF NOT EXISTS " + BLOCK_TABLE + " (" + WORLD_COLUMN + " VARCHAR(50), " + X_COLUMN + " INT, " + Y_COLUMN + " INT, " + Z_COLUMN + " INT, " + PLAYER_ID_COLUMN + " CHAR(36) NOT NULL, " + "PRIMARY KEY (" + WORLD_COLUMN + ", " + X_COLUMN + ", " + Y_COLUMN + ", " + Z_COLUMN + "));";
	}

	private String generateCreateEntityTableQuery() {
		return "CREATE TABLE IF NOT EXISTS " + ENTITY_TABLE + " (" + WORLD_COLUMN + " VARCHAR(50) NOT NULL, " + ENTITY_ID_COLUMN + " CHAR(36), " + PLAYER_ID_COLUMN + " CHAR(36) NOT NULL, " + "PRIMARY KEY (" + ENTITY_ID_COLUMN + "));";
	}

	@Override
	String generateGetOwnerQuery(Ownable ownable) {
		if (ownable instanceof OwnedBlock) {
			return generateGetOwnerQuery((OwnedBlock) ownable);
		} else if (ownable instanceof OwnedEntity) {
			return generateGetOwnerQuery((OwnedEntity) ownable);
		} else {
			throw new IllegalArgumentException("Unknown Ownable implementation");
		}
	}

	private String generateGetOwnerQuery(OwnedBlock block) {
		return "SELECT " + PLAYER_ID_COLUMN + " FROM " + BLOCK_TABLE + " WHERE " + WORLD_COLUMN + "='" + block.getWorldName() + "' AND " + X_COLUMN + "=" + block.getX() + " AND " + Y_COLUMN + "=" + block.getY() + " AND " + Z_COLUMN + "=" + block.getZ() + ";";
	}

	private String generateGetOwnerQuery(OwnedEntity entity) {
		return "SELECT " + PLAYER_ID_COLUMN + " FROM " + ENTITY_TABLE + " WHERE " + WORLD_COLUMN + "='" + entity.getWorldName() + "' AND " + ENTITY_ID_COLUMN + "='" + entity.getUniqueId().toString() + "';";
	}

	@Override
	String generateSetOwnerQuery(DatabaseAction databaseAction) {
		Ownable ownable = databaseAction.getOwnable();
		if (ownable instanceof OwnedBlock) {
			return generateSetBlockOwnerQuery(databaseAction);
		} else if (ownable instanceof OwnedEntity) {
			return generateSetEntityOwnerQuery(databaseAction);
		} else {
			throw new IllegalArgumentException("Unknown Ownable implementation");
		}
	}

	private String generateSetBlockOwnerQuery(DatabaseAction databaseAction) {
		OwnedBlock block = (OwnedBlock) databaseAction.getOwnable();
		String worldName = databaseAction.getOwnable().getWorldName();
		String playerId = databaseAction.getUser().getId().toString();
		int x = block.getX();
		int y = block.getY();
		int z = block.getZ();
		return "INSERT OR REPLACE INTO " + BLOCK_TABLE + "(" + WORLD_COLUMN + ", " + X_COLUMN + ", " + Y_COLUMN + ", " + Z_COLUMN + ", " + PLAYER_ID_COLUMN + ") VALUES('" + worldName + "', '" + x + "', '" + y + "', '" + z + "', '" + playerId + "');";
	}

	private String generateSetEntityOwnerQuery(DatabaseAction databaseAction) {
		OwnedEntity entity = (OwnedEntity) databaseAction.getOwnable();
		String worldName = databaseAction.getOwnable().getWorldName();
		String playerId = databaseAction.getUser().getId().toString();
		UUID entityId = entity.getUniqueId();
		return "INSERT OR REPLACE INTO " + ENTITY_TABLE + "(" + WORLD_COLUMN + ", " + ENTITY_ID_COLUMN + ", " + PLAYER_ID_COLUMN + ") VALUES('" + worldName + "', '" + entityId.toString() + "', '" + playerId + "');";
	}

	@Override
	String generateDeleteOwnerQuery(Ownable ownable) {
		if (ownable instanceof OwnedBlock) {
			return generateDeleteOwnerQuery((OwnedBlock) ownable);
		} else if (ownable instanceof OwnedEntity) {
			return generateDeleteOwnerQuery((OwnedEntity) ownable);
		} else {
			throw new IllegalArgumentException("Unknown Ownable implementation");
		}
	}

	private String generateDeleteOwnerQuery(OwnedEntity entity) {
		return "DELETE FROM " + ENTITY_TABLE + " WHERE " + WORLD_COLUMN + "='" + entity.getWorldName() + "' AND " + ENTITY_ID_COLUMN + "='" + entity.getUniqueId().toString() + "';";
	}

	private String generateDeleteOwnerQuery(OwnedBlock block) {
		return "DELETE FROM " + BLOCK_TABLE + " WHERE " + WORLD_COLUMN + "='" + block.getWorldName() + "' AND " + X_COLUMN + "=" + block.getX() + " AND " + Y_COLUMN + "=" + block.getY() + " AND " + Z_COLUMN + "=" + block.getZ() + ";";
	}

	@Override
	String[] generateDropUserQueries(User user) {
		String[] queries = new String[2];
		queries[0] = generateDropUserBlocksQuery(user);
		queries[1] = generateDropUserEntitiesQuery(user);
		return queries;
	}

	private String generateDropUserBlocksQuery(User user) {
		return "DELETE FROM " + BLOCK_TABLE + " WHERE " + PLAYER_ID_COLUMN + "='" + user.getId().toString() + "';";
	}

	private String generateDropUserEntitiesQuery(User user) {
		return "DELETE FROM " + ENTITY_TABLE + " WHERE " + PLAYER_ID_COLUMN + "='" + user.getId().toString() + "';";
	}

}
