package de.pheasn.blockown.database;

import de.pheasn.blockown.Output;
import de.pheasn.blockown.Ownable;
import de.pheasn.blockown.OwnedBlock;
import de.pheasn.blockown.OwnedEntity;
import de.pheasn.blockown.User;

import java.io.File;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.UUID;

public class MySqlDatabase extends CachedDatabase {

	private final String DATABASE_URL;

	/**
	 * Instantiates a new MySql database.
	 *
	 * @param output       the output
	 * @param pluginFolder the plugin folder
	 * @param host         the host
	 * @param port         the port
	 * @param databaseName the database name
	 * @param username     the username
	 * @param password     the password
	 * @throws SQLException           if connection fails
	 * @throws ClassNotFoundException if mysql driver class is missing
	 */
	public MySqlDatabase(Output output, File pluginFolder, String host, int port, String databaseName, String username, String password) throws SQLException, ClassNotFoundException {
		super(output, pluginFolder);
		Class.forName("com.mysql.jdbc.Driver");
		DATABASE_URL = "jdbc:mysql://" + host + ":" + port + "/" + databaseName + "?autoReconnect=true";
		connection = connect(username, password);
		createTables();
	}

	/**
	 * Instantiates a new MySql database without need for password. Not recommended.
	 *
	 * @param output       the output
	 * @param pluginFolder the plugin folder
	 * @param host         the host
	 * @param port         the port
	 * @param databaseName the database name
	 * @param username     the username
	 * @throws SQLException           if connection fails
	 * @throws ClassNotFoundException if mysql driver class is missing
	 */
	public MySqlDatabase(Output output, File pluginFolder, String host, int port, String databaseName, String username) throws SQLException, ClassNotFoundException {
		super(output, pluginFolder);
		Class.forName("com.mysql.jdbc.Driver");
		DATABASE_URL = "jdbc:mysql://" + host + ":" + port + "/" + databaseName + "?autoReconnect=true";
		connection = connect(username, null);
		createTables();
	}

	private Connection connect(String user, String password) throws SQLException {
		return DriverManager.getConnection(DATABASE_URL, user, password);
	}

	@Override
	String[] generateCreateTableQueries() {
		String[] queries = new String[2];
		queries[0] = generateCreateBlockTableQuery();
		queries[1] = generateCreateEntityTableQuery();
		return queries;
	}

	private String generateCreateBlockTableQuery() {
		return "CREATE TABLE IF NOT EXISTS " + BLOCK_TABLE + " (" + System.lineSeparator() + WORLD_COLUMN + " VARCHAR(50), " + System.lineSeparator() + X_COLUMN + " INT, " + System.lineSeparator() + Y_COLUMN + " INT, " + System.lineSeparator() + Z_COLUMN + " INT, " + System.lineSeparator() + PLAYER_ID_COLUMN + " CHAR(36) NOT NULL, " + System.lineSeparator() + "PRIMARY KEY (" + System.lineSeparator() + WORLD_COLUMN + ", " + System.lineSeparator() + X_COLUMN + ", " + System.lineSeparator() + Y_COLUMN + ", " + System.lineSeparator() + Z_COLUMN + "));";
	}

	private String generateCreateEntityTableQuery() {
		return "CREATE TABLE IF NOT EXISTS " + ENTITY_TABLE + " (" + WORLD_COLUMN + " VARCHAR(50) NOT NULL, " + ENTITY_ID_COLUMN + " CHAR(36), " + PLAYER_ID_COLUMN + " CHAR(36) NOT NULL, " + "PRIMARY KEY (" + ENTITY_ID_COLUMN + "));";
	}

	@Override
	String generateGetOwnerQuery(Ownable ownable) {
		if (ownable instanceof OwnedBlock) {
			return generateGetOwnerQuery((OwnedBlock) ownable);
		} else if (ownable instanceof OwnedEntity) {
			return generateGetOwnerQuery((OwnedEntity) ownable);
		} else {
			throw new IllegalArgumentException("Unknown ownable implementation");
		}
	}

	private String generateGetOwnerQuery(OwnedBlock block) {
		return "SELECT " + PLAYER_ID_COLUMN + " FROM " + BLOCK_TABLE + " WHERE " + WORLD_COLUMN + "='" + block.getWorldName() + "' AND " + X_COLUMN + "=" + block.getX() + " AND " + Y_COLUMN + "=" + block.getY() + " AND " + Z_COLUMN + "=" + block.getZ() + ";";
	}

	private String generateGetOwnerQuery(OwnedEntity entity) {
		return "SELECT " + PLAYER_ID_COLUMN + " FROM " + ENTITY_TABLE + " WHERE " + WORLD_COLUMN + "='" + entity.getWorldName() + "' AND " + ENTITY_ID_COLUMN + "='" + entity.getUniqueId().toString() + "';";
	}

	@Override
	String generateSetOwnerQuery(DatabaseAction databaseAction) {
		Ownable ownable = databaseAction.getOwnable();
		if (ownable instanceof OwnedBlock) {
			return generateSetBlockOwnerQuery(databaseAction);
		} else if (ownable instanceof OwnedEntity) {
			return generateSetEntityOwnerQuery(databaseAction);
		} else {
			throw new IllegalArgumentException("Unknown ownable implementation");
		}
	}

	private String generateSetBlockOwnerQuery(DatabaseAction databaseAction) {
		OwnedBlock block = (OwnedBlock) databaseAction.getOwnable();
		String worldName = databaseAction.getOwnable().getWorldName();
		String playerId = databaseAction.getUser().getId().toString();
		int x = block.getX();
		int y = block.getY();
		int z = block.getZ();
		return "INSERT INTO " + BLOCK_TABLE + "(" + WORLD_COLUMN + ", " + X_COLUMN + ", " + Y_COLUMN + ", " + Z_COLUMN + ", " + PLAYER_ID_COLUMN + ") VALUES('" + worldName + "', '" + x + "', '" + y + "', '" + z + "', '" + playerId + "')" + " ON DUPLICATE KEY UPDATE " + WORLD_COLUMN + "=" + "VALUES(" + WORLD_COLUMN + "), " + X_COLUMN + "=" + "VALUES(" + X_COLUMN + "), " + Y_COLUMN + "=" + "VALUES(" + Y_COLUMN + "), " + Z_COLUMN + "=" + "VALUES(" + Z_COLUMN + "), " + PLAYER_ID_COLUMN + "=" + "VALUES(" + PLAYER_ID_COLUMN + ");";
	}

	private String generateSetEntityOwnerQuery(DatabaseAction databaseAction) {
		OwnedEntity entity = (OwnedEntity) databaseAction.getOwnable();
		String worldName = databaseAction.getOwnable().getWorldName();
		String playerId = databaseAction.getUser().getId().toString();
		UUID entityId = entity.getUniqueId();
		return "INSERT INTO " + ENTITY_TABLE + "(" + WORLD_COLUMN + ", " + ENTITY_ID_COLUMN + ", " + PLAYER_ID_COLUMN + ") VALUES('" + worldName + "', '" + entityId.toString() + "', '" + playerId + "') ON DUPLICATE KEY UPDATE " + WORLD_COLUMN + "=" + "VALUES(" + WORLD_COLUMN + "), " + ENTITY_ID_COLUMN + "=" + "VALUES(" + ENTITY_ID_COLUMN + "), " + PLAYER_ID_COLUMN + "=" + "VALUES(" + PLAYER_ID_COLUMN + ");";
	}

	@Override
	String generateDeleteOwnerQuery(Ownable ownable) {
		if (ownable instanceof OwnedBlock) {
			return generateDeleteOwnerQuery((OwnedBlock) ownable);
		} else if (ownable instanceof OwnedEntity) {
			return generateDeleteOwnerQuery((OwnedEntity) ownable);
		} else {
			throw new IllegalArgumentException("Unknown Ownable implementation");
		}
	}

	private String generateDeleteOwnerQuery(OwnedEntity entity) {
		String worldName = entity.getWorldName();
		return "DELETE FROM " + ENTITY_TABLE + " WHERE " + WORLD_COLUMN + "='" + worldName + "' AND " + ENTITY_ID_COLUMN + "='" + entity.getUniqueId().toString() + "';";
	}

	private String generateDeleteOwnerQuery(OwnedBlock block) {
		String worldName = block.getWorldName();
		return "DELETE FROM " + BLOCK_TABLE + " WHERE " + WORLD_COLUMN + "='" + worldName + "' AND " + X_COLUMN + "=" + block.getX() + " AND " + Y_COLUMN + "=" + block.getY() + " AND " + Z_COLUMN + "=" + block.getZ() + ";";
	}

	@Override
	String[] generateDropUserQueries(User user) {
		String[] queries = new String[2];
		queries[0] = generateDropUserBlocksQuery(user);
		queries[1] = generateDropUserEntitiesQuery(user);
		return queries;
	}

	private String generateDropUserBlocksQuery(User user) {
		return "DELETE FROM " + BLOCK_TABLE + " WHERE " + PLAYER_ID_COLUMN + "='" + user.getId().toString() + "';";
	}

	private String generateDropUserEntitiesQuery(User user) {
		return "DELETE FROM " + ENTITY_TABLE + " WHERE " + PLAYER_ID_COLUMN + "='" + user.getId().toString() + "';";
	}
}
