package de.pheasn.blockown.database;

import de.pheasn.blockown.Output;
import de.pheasn.blockown.Ownable;
import de.pheasn.blockown.User;

import java.io.File;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.TimeUnit;

public abstract class CachedDatabase extends Database {

	final CacheAccessor cache;

	CachedDatabase(Output output, File pluginFolder) {
		super(output);
		cache = new CacheAccessor(this, pluginFolder);
	}

	@Override
	public User getOwner(Ownable ownable) {
		return cache.getOwner(ownable);
	}

	@Override
	public void enqueue(DatabaseAction databaseAction) {
		try {
			threadPool.execute(new CachedDatabaseOperation(this, databaseAction));
		} catch (RejectedExecutionException ignored) {
		}
	}

	@Override
	public void run() {
		while (!disable) {
			cache.flush();
			try {
				Thread.sleep(1000);
			} catch (InterruptedException ignored) {
			}
		}

		threadPool.shutdown();
		getOutput().debugMessage("Waiting for database actions termination");
		try {
			threadPool.awaitTermination(10, TimeUnit.SECONDS);
		} catch (InterruptedException ignored) {
		}

		// Flush cache before thread is ending
		cache.flush();
		close();
	}
}

class CachedDatabaseOperation implements Runnable {

	private final DatabaseAction action;
	private final CachedDatabase database;

	CachedDatabaseOperation(CachedDatabase database, DatabaseAction action) {
		this.action = action;
		this.database = database;
	}

	@Override
	public void run() {
		if (!database.cache.doAction(action))
			database.getOutput().printException("An action has not been performed", new ActionNotPerformedException(action));
	}

}
