package de.pheasn.blockown;

import org.bukkit.Bukkit;
import org.bukkit.World;
import org.bukkit.entity.Entity;
import org.bukkit.entity.EntityType;

import java.lang.ref.WeakReference;
import java.util.UUID;

public class OwnedEntity implements Ownable {

	private static final long serialVersionUID = -5197934920645104143L;

	private transient WeakReference<Entity> entityRef = new WeakReference<>(null);

	private final UUID id;
	private final String worldName;
	private transient Material material = null;

	/**
	 * Creates a new OwnedEntity instance holding a WeakReference to the given block.
	 *
	 * @param entity
	 *            the entity
	 * @return the new OwnedEntity instance
	 */
	public static OwnedEntity newInstance(Entity entity) {
		if(entity == null) throw new NullPointerException("Entity can't be null");
		return new OwnedEntity(entity.getUniqueId(), entity.getWorld().getName(), new Material(entity.getType()), entity);
	}

	/**
	 * Creates a new OwnedEntity instance. The corresponding Entity will be looked up when getEntity() is called.
	 *
	 * @param id
	 *            the UUID
	 * @param worldName
	 *            the worldName
	 * @param material
	 *            the material
	 * @return the new OwnedEntity instance
	 */
	public static OwnedEntity newInstance(UUID id, String worldName, Material material) {
		if(id == null) throw new NullPointerException("UUID can't be null");
		if(worldName == null) throw new NullPointerException("world name can't be null");
		// it's okay if material is null
		return new OwnedEntity(id, worldName, material, null);
	}

	/**
	 * Creates a new OwnedEntity instance. The corresponding Entity and its Material will be looked up when needed.
	 *
	 * @param id
	 *            the UUID
	 * @param worldName
	 *            the worldName
	 * @return the new OwnedEntity instance
	 */
	public static OwnedEntity newInstance(UUID id, String worldName) {
		return newInstance(id, worldName, null);
	}

	private OwnedEntity(UUID id, String worldName, Material material, Entity entity) {
		this.id = id;
		this.worldName = worldName;
		this.material = material;
		this.entityRef = new WeakReference<>(entity);
	}

	/**
	 * Gets the entity corresponding to this OwnedEntity. Keep in mind that this can be a very expensive operation.
	 *
	 * @return the entity
	 * @throws InvalidWorldException
	 *             if the world name is invalid
	 * @throws InvalidUUIDException
	 *             if the entity can't be found in the specified world
	 */
	public Entity getEntity() throws InvalidWorldException, InvalidUUIDException {
		if (entityRef == null) entityRef = new WeakReference<>(null);
		Entity result = entityRef.get();
		if (result == null) {
			for (Entity entity : getWorld().getEntitiesByClass(material.getEntityType().getEntityClass())) {
				if (entity.getUniqueId().equals(id)) {
					entityRef = new WeakReference<>(entity);
					return entity;
				}
			}
			throw new InvalidUUIDException("No entity with UUID " + id + " could be found in world " + worldName);
		} else {
			return result;
		}
	}

	public UUID getUniqueId() {
		return id;
	}

	@Override
	public World getWorld() throws InvalidWorldException {
		World world = Bukkit.getServer().getWorld(worldName);
		if (world == null) throw new InvalidWorldException("There is no world called " + worldName);
		return world;
	}

	@Override
	public String getWorldName() {
		return worldName;
	}

	@Override
	public Material getMaterial() throws InvalidWorldException {
		if (material == null) {
			material = new Material(getEntity().getType());
		}
		return material;
	}

	/**
	 * Checks if the entity is living. This can be a very expensive operation if it needs to look up the entity.
	 *
	 * @return true, if it is a LivingEntity
	 * @throws InvalidWorldException
	 *             if the world name is invalid
	 * @throws InvalidUUIDException
	 *             if the entity can't be found in the specified world
	 */
	public boolean isLiving() throws InvalidWorldException, InvalidUUIDException {
		EntityType type = getMaterial().getEntityType();
		return type.isAlive();
	}

	@Override
	public int hashCode() {
		return id.hashCode();
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj) return true;
		if (!(obj instanceof OwnedEntity)) return false;
		OwnedEntity other = (OwnedEntity) obj;
		if (!id.equals(other.id)) return false;
		return true;
	}

	@Override
	public String toString() {
		return "OwnedEntity [id=" + id + ", worldName=" + worldName + ", material=" + material + "]";
	}

}
