package de.pheasn.blockown;

import de.pheasn.blockown.command.*;
import de.pheasn.blockown.database.Database;
import de.pheasn.blockown.database.MySqlDatabase;
import de.pheasn.blockown.database.SqliteDatabase;
import de.pheasn.blockown.event.L_BlockBreak;
import de.pheasn.blockown.event.L_BlockPlace;
import de.pheasn.blockown.event.L_EnvironmentDamage;
import de.pheasn.blockown.event.L_PlayerInteract;
import de.pheasn.blockown.protection.Protection;
import de.pheasn.pluginupdater.*;
import org.bukkit.Bukkit;
import org.bukkit.World;
import org.bukkit.entity.Animals;
import org.bukkit.entity.LivingEntity;
import org.bukkit.plugin.java.JavaPlugin;
import org.mcstats.Metrics;
import org.mcstats.Metrics.Graph;
import org.mcstats.Metrics.Plotter;

import java.io.File;
import java.io.IOException;
import java.sql.SQLException;
import java.util.*;

public class BlockOwn extends JavaPlugin implements Updatable {

	private static final String DATA_FOLDER_PATH = "plugins" + File.separator + "BlockOwn";
	private static final int PLUGIN_ID = 62749;
	private static final ReleaseChannel RELEASE_CHANNEL = ReleaseChannel.STABLE;

	private Output output;

	private Database database;
	private Thread databaseThread;

	private Protection protection;
	private Thread protectionThread;

	private Updater updater;

	private Set<String> enabledWorlds;
	private Set<Material> ownEnabledMaterials;
	private Set<Material> ownDisabledMaterials;

	private File messageFile;
	private File configFile;
	private File protectionFile;
	private File importerFolder;

	private Set<User> ignoringUsers;
	private Set<User> nonOwningUsers;
	private Map<User, WaitType> waitingUsers;

	@Override
	public void onDisable() {
		if (database != null) database.disable();
		if (protection != null) protection.disable();
		if (databaseThread != null && protectionThread != null) {
			while (databaseThread.isAlive() || protectionThread.isAlive()) {
				try {
					Thread.sleep(1000);
				} catch (InterruptedException ignored) {
				}
			}
		}

		super.onDisable();
	}

	@Override
	public void onEnable() {
		super.onEnable();
		configFile = new File(getDataFolder(), "config.yml");
		Setting.setConfig(getConfig(), configFile);
		output = Setting.DEBUG_MODE.get() ? new DebugOutput() : new DefaultOutput();
		output.debugMessage("Debug mode enabled");
		Setting.setOutput(output);
		User.initialize(this);

		messageFile = new File(getDataFolder(), "messages.yml");
		protectionFile = new File(getDataFolder(), "protection.dat");
		importerFolder = new File(getDataFolder(), "Importer");


		if (!createDataStructure()) {
			getOutput().printError("Error creating data structure.");
			Bukkit.getPluginManager().disablePlugin(this);
			return;
		}

		updateDeprecatedSettings();

		if (!initializeDatabase()) {
			getOutput().printError("Couldn't connect to database.");
			Bukkit.getPluginManager().disablePlugin(this);
			return;
		}

		if (!initializeProtection()) {
			getOutput().printError("Couldn't initialize Protection.");
			Bukkit.getPluginManager().disablePlugin(this);
			return;
		}

		enabledWorlds = getEnabledWorlds();
		ownEnabledMaterials = getOwnEnabledMaterials();
		ownDisabledMaterials = getOwnDisabledMaterials();
		ignoringUsers = new HashSet<>(getServer().getOfflinePlayers().length * 2);
		nonOwningUsers = new HashSet<>(getServer().getOfflinePlayers().length * 2);
		waitingUsers = new HashMap<>(getServer().getOfflinePlayers().length * 2);

		registerListeners();
		registerCommands();

		if(Setting.UPDATER_ENABLE.get()) {
			String apiKey = Setting.UPDATER_API_KEY.get() == null ? null : Setting.UPDATER_API_KEY.get().toString();
			updater = new Updater(this, PLUGIN_ID, 30 * 60 * 1000, getFile(), apiKey, Setting.UPDATER_RELEASE_CHANNEL.getReleaseChannel());
			updater.start();
		}

		initializeMetrics();

		Setting.VERSION.set(this.getDescription(UpdaterKey.VERSION) + "-" + getReleaseChannel());
	}

	@SuppressWarnings("deprecation")
	private void updateDeprecatedSettings() {
		PluginVersion oldVersion = new PluginVersion(Setting.VERSION.get().toString(), getReleaseChannel());
		for (Setting.OLD oldSetting : Setting.OLD.values()) {
			PluginVersion deprecatingVersion = new PluginVersion(oldSetting.getDeprecatingVersion(), getReleaseChannel());
			if (deprecatingVersion.compareTo(oldVersion) > 0) {
				oldSetting.transferValue();
				oldSetting.getSetting().set(null);
			}
		}
	}

	private void initializeMetrics() {
		try {
			Metrics metrics = new Metrics(this);
			addDatabaseGraph(metrics);
			addOfflinePlayerCountGraph(metrics);
			metrics.start();
		} catch (IOException e) {
			// Failed to submit the stats
			getOutput().debugMessage("Failed to submit stats, nothing to worry about");
		}
	}

	private void addDatabaseGraph(Metrics metrics) {
		Graph databaseGraph = metrics.createGraph("Type of database used");
		databaseGraph.addPlotter(new Plotter("Sqlite") {
			@Override
			public int getValue() {
				return getOwningDatabase() instanceof SqliteDatabase ? 1 : 0;
			}
		});
		databaseGraph.addPlotter(new Plotter("MySQL") {
			@Override
			public int getValue() {
				return getOwningDatabase() instanceof MySqlDatabase ? 1 : 0;
			}
		});
	}

	private void addOfflinePlayerCountGraph(Metrics metrics) {
		Graph databaseGraph = metrics.createGraph("Offline Player count");
		final int offlinePlayerCount = getServer().getOfflinePlayers().length;
		databaseGraph.addPlotter(new Plotter("0-10") {
			public int getValue() {
				return offlinePlayerCount <= 10 ? 1 : 0;
			}
		});
		databaseGraph.addPlotter(new Plotter("11-50") {
			public int getValue() {
				return offlinePlayerCount > 10 && offlinePlayerCount <= 50 ? 1 : 0;
			}
		});
		databaseGraph.addPlotter(new Plotter("51-100") {
			public int getValue() {
				return offlinePlayerCount > 50 && offlinePlayerCount <= 100 ? 1 : 0;
			}
		});
		databaseGraph.addPlotter(new Plotter("101-1000") {
			public int getValue() {
				return offlinePlayerCount > 100 && offlinePlayerCount <= 1000 ? 1 : 0;
			}
		});
		databaseGraph.addPlotter(new Plotter(">1000") {
			public int getValue() {
				return offlinePlayerCount > 1000 ? 1 : 0;
			}
		});
	}

	private Set<String> getEnabledWorlds() {
		HashSet<String> result = new HashSet<>(20);
		List<String> disabled = Setting.DISABLED_WORLDS.get();
		List<String> enabled = Setting.ENABLED_WORLDS.get();

		if (disabled.isEmpty() && enabled.isEmpty()) {
			for (World world : getServer().getWorlds()) {
				result.add(world.getName());
			}
			return result;
		}

		Set<String> disabledLower = new HashSet<>(disabled.size());
		for (String s : disabled)
			disabledLower.add(s.toLowerCase());
		Set<String> enabledLower = new HashSet<>(enabled.size());
		for (String s : enabled)
			enabledLower.add(s.toLowerCase());

		for (World world : getServer().getWorlds()) {
			if (disabledLower.contains(world.getName().toLowerCase())) continue;
			if (enabledLower.isEmpty() || enabledLower.contains(world.getName().toLowerCase())) {
				result.add(world.getName());
			}
		}
		return result;
	}

	private Set<Material> getOwnEnabledMaterials() {
		List<String> enabled = Setting.DATABASE_ALLOWED_MATERIALS.get();
		HashSet<Material> result = new HashSet<>(enabled.size(), 1);
		Material m;
		for (String materialName : enabled) {
			try {
				m = Material.parseMaterial(materialName);
				result.add(m);
			} catch (IllegalArgumentException ignored) {
			}
		}
		return result;
	}

	private Set<Material> getOwnDisabledMaterials() {
		List<String> disabled = Setting.DATABASE_DISALLOWED_MATERIALS.get();
		HashSet<Material> result = new HashSet<>(disabled.size(), 1);
		Material m;
		for (String materialName : disabled) {
			try {
				m = Material.parseMaterial(materialName);
				result.add(m);
			} catch (IllegalArgumentException ignored) {
			}
		}
		return result;
	}

	public boolean isOwnEnabled(Material material) {
		if (ownDisabledMaterials.contains(material)) return false;
		if (ownEnabledMaterials.contains(material)) return true;
		if (material.isBlock()) return ownEnabledMaterials.isEmpty();
		if (Animals.class.isAssignableFrom(material.getEntityType().getEntityClass())) {
			return Setting.DATABASE_ALLOW_ANIMALS.get();
		}
		if (!LivingEntity.class.isAssignableFrom(material.getEntityType().getEntityClass())) {
			return Setting.DATABASE_ALLOW_ENTITIES.get();
		}
		return false;
	}

	public boolean isIgnoring(User user) {
		return ignoringUsers.contains(user);
	}

	public void setIgnoring(User user, boolean enable) {
		if (enable) {
			ignoringUsers.add(user);
		} else {
			ignoringUsers.remove(user);
		}
	}

	public boolean isOwning(User user) {
		return !nonOwningUsers.contains(user);
	}

	public void setOwning(User user, boolean enable) {
		if (enable) {
			nonOwningUsers.remove(user);
		} else {
			nonOwningUsers.add(user);
		}
	}

	public void addWaiting(User user, WaitType waitType) {
		waitingUsers.put(user, waitType);
	}

	public boolean isWaiting(User user, WaitType waitType) {
		return waitingUsers.get(user) == waitType;
	}

	public WaitType getWaiting(User user) {
		return waitingUsers.get(user);
	}

	public void removeWaiting(User user) {
		waitingUsers.remove(user);
	}

	private boolean createDataStructure() {
		File dataFolder = getDataFolder();
		if (!dataFolder.exists()) {
			if (!dataFolder.mkdir()) return false;
		}
		getConfig().options().copyDefaults(true);
		if (!configFile.exists()) saveDefaultConfig();
		if (!Message.loadMessages(this)) return false;
		if (!importerFolder.exists()) {
			if (!importerFolder.mkdir()) return false;
		}
		return true;
	}

	private boolean initializeDatabase() {
		if (Setting.DATABASE_MYSQL_ENABLE.get()) {
			try {
				if (Setting.DATABASE_MYSQL_PASSWORD.get().equals("")) {
					database = new MySqlDatabase(getOutput(),
							getDataFolder(),
							Setting.DATABASE_MYSQL_HOST.get().toString(),
							Setting.DATABASE_MYSQL_PORT.get(),
							Setting.DATABASE_MYSQL_DATABASE.get().toString(),
							Setting.DATABASE_MYSQL_USER.get().toString());
				} else {
					database = new MySqlDatabase(getOutput(),
							getDataFolder(),
							Setting.DATABASE_MYSQL_HOST.get().toString(),
							Setting.DATABASE_MYSQL_PORT.get(),
							Setting.DATABASE_MYSQL_DATABASE.get().toString(),
							Setting.DATABASE_MYSQL_USER.get().toString(),
							Setting.DATABASE_MYSQL_PASSWORD.get().toString());
				}
				databaseThread = new Thread(database, "DatabaseThread");
				databaseThread.start();
				return true;
			} catch (ClassNotFoundException e) {
				getOutput().printError("MySQL driver class not found", e);
				return false;
			} catch (SQLException e) {
				getOutput().printError("Exception during MySQL initialization", e);
				return false;
			}
		} else {
			try {
				database = new SqliteDatabase(getOutput(), getDataFolder());
				databaseThread = new Thread(database, "DatabaseThread");
				databaseThread.start();
				return true;
			} catch (ClassNotFoundException e) {
				getOutput().printError("SQLite driver class not found", e);
				return false;
			} catch (SQLException e) {
				getOutput().printError("Exception during SQLite initialization", e);
				return false;
			}
		}
	}

	private boolean initializeProtection() {
		try {
			protection = new Protection(getOutput(), getProtectionFile());
			protectionThread = new Thread(protection, "ProtectionThread");
			protectionThread.start();
			return true;
		} catch (IOException e) {
			getOutput().printError("Exception while loading protections file", e);
			return false;
		}
	}

	private void registerListeners() {
		getServer().getPluginManager().registerEvents(new L_BlockPlace(this), this);
		getServer().getPluginManager().registerEvents(new L_EnvironmentDamage(this), this);
		getServer().getPluginManager().registerEvents(new L_PlayerInteract(this), this);
		getServer().getPluginManager().registerEvents(new L_BlockBreak(this), this);
	}

	private void registerCommands() {
		getCommand(Command.IGNORE.toString()).setExecutor(new CE_Ignore(this));
		getCommand(Command.IGNORE.toString()).setUsage("/<command> [on | off]");
		getCommand(Command.IMPORT.toString()).setExecutor(new CE_Import(this));
		getCommand(Command.IMPORT.toString()).setUsage("/<command> <importer-name>");
		getCommand(Command.SHOWOWNER.toString()).setExecutor(new CE_ShowOwner(this));
		getCommand(Command.SHOWOWNER.toString()).setUsage("/<command> [e | b]");
		getCommand(Command.ADDFRIEND.toString()).setExecutor(new CE_AddFriend(this));
		getCommand(Command.ADDFRIEND.toString()).setUsage("/<command> <player>");
		getCommand(Command.REMFRIEND.toString()).setExecutor(new CE_RemFriend(this));
		getCommand(Command.REMFRIEND.toString()).setUsage("/<command> <player>");
		getCommand(Command.LOCK.toString()).setExecutor(new CE_Lock(this));
		getCommand(Command.LOCK.toString()).setUsage("/<command> [e | b | <material-name>]");
		getCommand(Command.UNLOCK.toString()).setExecutor(new CE_Unlock(this));
		getCommand(Command.UNLOCK.toString()).setUsage("/<command> [e | b | <material-name>]");
		getCommand(Command.PROTECT.toString()).setExecutor(new CE_Protect(this));
		getCommand(Command.PROTECT.toString()).setUsage("/<command> [e | b | <material-name>]");
		getCommand(Command.UNPROTECT.toString()).setExecutor(new CE_Unprotect(this));
		getCommand(Command.UNPROTECT.toString()).setUsage("/<command> [e | b | <material-name>]");
		getCommand(Command.OWN.toString()).setExecutor(new CE_Own(this));
		getCommand(Command.OWN.toString()).setUsage("/<command> [e | b]");
		getCommand(Command.UNOWN.toString()).setExecutor(new CE_Unown(this));
		getCommand(Command.UNOWN.toString()).setUsage("/<command> [e | b]");
		getCommand(Command.LIST.toString()).setExecutor(new CE_List(this));
		getCommand(Command.LIST.toString()).setUsage("/<command> protected | locked | friends");
		getCommand(Command.UNPROTECTPLAYER.toString()).setExecutor(new CE_UnprotectPlayer(this));
		getCommand(Command.UNPROTECTPLAYER.toString()).setUsage("/<command> <player>");
		getCommand(Command.UNOWNPLAYER.toString()).setExecutor(new CE_UnownPlayer(this));
		getCommand(Command.UNOWNPLAYER.toString()).setUsage("/<command> <player>");
		getCommand(Command.OWNING.toString()).setExecutor(new CE_Owning(this));
		getCommand(Command.OWNING.toString()).setUsage("/<command> [on | off]");
	}

	public Database getOwningDatabase() {
		return database;
	}

	public Protection getProtection() {
		return protection;
	}

	public boolean isEnabledInWorld(World world) {
		return enabledWorlds.contains(world.getName());
	}

	@Override
	public void updateIOException(IOException e) {
		getOutput().printException("Error while updating", e);
	}

	@Override
	public void updateInterruptedException(InterruptedException e) {
		getOutput().printException("Error while updating", e);
	}

	@Override
	public String getDescription(UpdaterKey key) {
		switch (key) {
			case NAME:
				return getDescription().getName();
			case MAIN_AUTHOR:
				return getDescription().getAuthors().get(0);
			case VERSION:
				return getDescription().getVersion();
			default:
				return null;
		}
	}

	@Override
	public void updateNotifyDownloading() {
		getOutput().printConsole(Message.UPDATER_DOWNLOADING);
	}

	@Override
	public void updateNotifyRestarting() {
		getOutput().printConsole(Message.UPDATER_RESTART);
	}

	@Override
	public void updateNotifyReadUpdatedProjectPage() {
		getOutput().printConsole(Message.UPDATER_BIG_UPDATE);
	}

	@Override
	public ReleaseChannel getReleaseChannel() {
		return RELEASE_CHANNEL;
	}

	public static String getDataFolderPath() {
		return DATA_FOLDER_PATH;
	}

	public Output getOutput() {
		return output;
	}

	public File getMessageFile() {
		return messageFile;
	}

	public File getProtectionFile() {
		return protectionFile;
	}

	public File getImporterFolder() {
		return importerFolder;
	}
}
