package de.pheasn.blockown.protection;

import de.pheasn.blockown.AbstractBuilder;
import de.pheasn.blockown.Material;
import de.pheasn.blockown.User;

import java.text.SimpleDateFormat;
import java.util.Date;

public class ProtectAction {

	private final Type type;
	private final User owner;
	private final User user;
	private final Material material;
	private final long timeStamp;

	public enum Type {
		PROTECT, UNPROTECT, LOCK, UNLOCK, FRIEND, UNFRIEND, DROP
	}

	public static class Builder extends AbstractBuilder<ProtectAction> {

		private Type type;
		private User owner;
		private Material material;
		private User user;

		public Builder(User owner) {
			this.owner = owner;
		}

		public Builder protect(Material material) {
			this.material = material;
			type = Type.PROTECT;
			return this;
		}

		public Builder unprotect(Material material) {
			this.material = material;
			type = Type.UNPROTECT;
			return this;
		}

		public Builder lock(Material material) {
			this.material = material;
			type = Type.LOCK;
			return this;
		}

		public Builder unlock(Material material) {
			this.material = material;
			type = Type.UNLOCK;
			return this;
		}

		public Builder friend(User user) {
			this.user = user;
			type = Type.FRIEND;
			return this;
		}

		public Builder unfriend(User user) {
			this.user = user;
			type = Type.UNFRIEND;
			return this;
		}

		public Builder drop() {
			type = Type.DROP;
			return this;
		}

		@Override
		public ProtectAction build() {
			return new ProtectAction(type, owner, material, user);
		}

	}
	/**
	 * Instantiates a new protect action.
	 *
	 * @param type				the type of action
	 * @param owner             Must not be null
	 * @param material          may be null or left out depending on action type
	 * @param user              may be null or left out depending on action type
	 */
	private ProtectAction(Type type, User owner, Material material, User user) {
		//TODO

		switch(type) {
		case DROP:
			break;
		case FRIEND:
		case UNFRIEND:
			if(user == null)
				throw new NullPointerException("User can't be null");
			if(user.isNobody())
				throw new IllegalArgumentException("User can't be nobody");
			break;
		case LOCK:
		case UNLOCK:
		case PROTECT:
		case UNPROTECT:
			if(material == null)
				throw new NullPointerException("Material can't be null");
			break;
		default:
			throw new IllegalArgumentException("Invalid ProtectAction.Type");
		}

		if(owner == null)
			throw new NullPointerException("Owner can't be null");
		if(owner.isNobody())
			throw new IllegalArgumentException("Owner can't be nobody");

		this.type = type;
		this.owner = owner;
		this.material = material;
		this.user = user;
		this.timeStamp = System.currentTimeMillis();
	}

	public Type getActionType() {
		return type;
	}

	public User getOwner() {
		return owner;
	}

	public User getUser() {
		return user;
	}

	public Material getMaterial() {
		return material;
	}

	public long getTimeStamp() {
		return timeStamp;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 17;
		result = prime * result + ((material == null) ? 0 : material.hashCode());
		result = prime * result + ((owner == null) ? 0 : owner.hashCode());
		result = prime * result + ((type == null) ? 0 : type.hashCode());
		result = prime * result + (int) (timeStamp ^ (timeStamp >>> 32));
		result = prime * result + ((user == null) ? 0 : user.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj) return true;
		if (!(obj instanceof ProtectAction)) return false;
		ProtectAction other = (ProtectAction) obj;
		if (material == null) {
			if (other.material != null) return false;
		} else if (!material.equals(other.material)) return false;
		if (owner == null) {
			if (other.owner != null) return false;
		} else if (!owner.equals(other.owner)) return false;
		if (type != other.type) return false;
		if (timeStamp != other.timeStamp) return false;
		if (user == null) {
			if (other.user != null) return false;
		} else if (!user.equals(other.user)) return false;
		return true;
	}

	@Override
	public String toString() {
		return "ProtectAction [protectActionType=" + type + ", owner=" + owner + ", user=" + user + ", material=" + material + ", timeStamp="
				+ new SimpleDateFormat().format(new Date(timeStamp)) + "]";
	}
}
