package de.pheasn.blockown.importer;

import com.evilmidget38.UUIDFetcher;
import de.pheasn.blockown.User;
import de.pheasn.blockown.database.Database;
import org.bukkit.Bukkit;
import org.bukkit.Location;
import org.bukkit.World;
import org.bukkit.entity.Entity;
import org.bukkit.entity.EntityType;
import org.bukkit.entity.ItemFrame;

import java.io.File;
import java.util.*;

public abstract class Importer implements Runnable {

	public static boolean debug = false;

	protected final Database db;
	protected final File pluginsFolder;

	private final Runnable runnable;
	private final HashMap<String, UUID> knownUUIDs;

	/**
	 * Instantiates a new Importer
	 *
	 * @param db            The destination database
	 * @param pluginsFolder the "plugins" folder
	 * @param runnable      the runnable to run when the import has finished
	 */
	protected Importer(Database db, File pluginsFolder, Runnable runnable) {
		this.db = db;
		this.pluginsFolder = pluginsFolder;
		this.runnable = runnable;
		this.knownUUIDs = new HashMap<>(1024);
	}

	/**
	 * Import all database entries from the source into the given database. Don't overwrite existing entries.
	 *
	 * <b>This also needs to call the finished() method when successfully done.</b>
	 */
	@Override
	public abstract void run();

	/**
	 * Needs to be called when successfully finished.
	 */
	protected void finished() {
		runnable.run();
	}

	protected HashMap<UUID, EntityType> getEntities() {
		synchronized (Bukkit.getServer()) {
			HashMap<UUID, EntityType> entities = new HashMap<>(2048);
			for (World world : Bukkit.getServer().getWorlds()) {
				for (Entity entity : world.getEntities()) {
					entities.put(entity.getUniqueId(), entity.getType());
				}
			}
			return entities;
		}
	}

	@SuppressWarnings("deprecation")
	protected User getUser(String name) {
		if (knownUUIDs.containsKey(name)) {
			UUID known = knownUUIDs.get(name);
			if (known != null) return User.getInstance(known);
			else return null;
		}

		UUID id;
		synchronized (Bukkit.getServer()) {
			id = Bukkit.getServer().getOfflinePlayer(name).getUniqueId();
		}
		if (id != null) {
			knownUUIDs.put(name, id);
			return User.getInstance(id);
		} else {
			UUIDFetcher fetcher = new UUIDFetcher(Collections.singletonList(name));
			try {
				id = fetcher.call().get(name);
			} catch (Exception ignored) {
			}
			knownUUIDs.put(name, id);
			if (id != null) {
				return User.getInstance(id);
			} else {
				return null;
			}
		}

	}

	protected Map<String, UUID> getUsers(List<String> names) throws Exception {
		UUIDFetcher fetcher = new UUIDFetcher(names);
		return fetcher.call();
	}

	protected ItemFrame getItemFrame(String worldName, int x, int y, int z) {
		synchronized (Bukkit.getServer()) {
			World world = Bukkit.getServer().getWorld(worldName);
			if (world == null) return null;
			Location loc = new Location(world, x, y, z);
			for (Entity entity : loc.getChunk().getEntities()) {
				if (entity instanceof ItemFrame) {
					if (entity.getLocation().getBlockX() == loc.getBlockX() && entity.getLocation().getBlockY() == loc.getBlockY()
							&& entity.getLocation().getBlockZ() == loc.getBlockZ()) {
						return (ItemFrame) entity;
					}
				}
			}
			return null;
		}
	}
}
