package de.pheasn.blockown.database;

import de.pheasn.blockown.Ownable;
import de.pheasn.blockown.User;

import java.text.SimpleDateFormat;
import java.util.Date;

public class DatabaseAction {

	private final User user;
	private final Ownable ownable;
	private final Type type;
	private final long timeStamp;
	private final int hashCode;

	public enum Type {
		DROP, OWN, UNOWN
	}

	public static DatabaseAction newUnownInstance(Ownable ownable) {
		if (ownable == null)
			throw new NullPointerException("Ownable to unown must be specified");
		return new DatabaseAction(Type.UNOWN, ownable, null);
	}

	public static DatabaseAction newOwnInstance(Ownable ownable, User user) {
		if (user == null)
			throw new NullPointerException("User to own for must be specified");
		if (user.isNobody())
			throw new IllegalArgumentException(
					"User to own for can't be nobody");
		if (ownable == null)
			throw new NullPointerException("Ownable to own must be specified");
		return new DatabaseAction(Type.OWN, ownable, user);
	}

	public static DatabaseAction newDropInstance(User user) {
		if (user == null)
			throw new NullPointerException("User to drop can't be null");
		if (user.isNobody())
			throw new IllegalArgumentException("User to drop can't be nobody");
		return new DatabaseAction(Type.DROP, null, user);
	}

	/**
	 * Instantiates a new DatabaseAction.
	 *
	 * @param type
	 *            the DatabaseActionType
	 * @param ownable
	 *            the ownable, may be null if type is DROP
	 * @param user
	 *            the user, may be null if type is UNOWN
	 */
	private DatabaseAction(Type type, Ownable ownable, User user) {
		this.user = user;
		this.ownable = ownable;
		this.type = type;
		this.timeStamp = System.currentTimeMillis();
		this.hashCode = calcHashCode();
	}

	public User getUser() {
		return user;
	}

	public Ownable getOwnable() {
		return ownable;
	}

	public long getTimeStamp() {
		return timeStamp;
	}

	public Type getActionType() {
		return type;
	}

	private int calcHashCode() {
		final int prime = 31;
		int result = 3;
		result = prime * result + ((type == null) ? 0 : type.hashCode());
		result = prime * result + ((ownable == null) ? 0 : ownable.hashCode());
		result = prime * result + (int) (timeStamp ^ (timeStamp >>> 32));
		result = prime * result + ((user == null) ? 0 : user.hashCode());
		return result;
	}

	@Override
	public int hashCode() {
		return hashCode;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (!(obj instanceof DatabaseAction))
			return false;
		DatabaseAction other = (DatabaseAction) obj;
		if (type != other.type)
			return false;
		if (ownable == null) {
			if (other.ownable != null)
				return false;
		} else if (!ownable.equals(other.ownable))
			return false;
		if (Long.compare(timeStamp, other.timeStamp) != 0)
			return false;
		if (user == null) {
			if (other.user != null)
				return false;
		} else if (!user.equals(other.user))
			return false;
		return true;
	}

	@Override
	public String toString() {
		return "DatabaseAction [user=" + user + ", ownable=" + ownable
				+ ", type=" + type + ", timeStamp="
				+ new SimpleDateFormat().format(new Date(timeStamp)) + "]";
	}
}
