package de.pheasn.blockown.database;

import de.pheasn.blockown.*;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.TimeUnit;

public abstract class Database implements Runnable {

	// Column names
	static final String PLAYER_ID_COLUMN = "player_id";
	static final String WORLD_COLUMN = "world";
	static final String ENTITY_ID_COLUMN = "entity_id";
	static final String X_COLUMN = "x";
	static final String Y_COLUMN = "y";
	static final String Z_COLUMN = "z";

	// Table names
	static final String BLOCK_TABLE = "block_table";
	static final String ENTITY_TABLE = "entity_table";

	final ExecutorService threadPool;
	volatile boolean disable = false;

	final Output output;

	Connection connection;

	Database(Output output) {
		this.output = output;
		this.threadPool = Executors.newCachedThreadPool();
	}

	Output getOutput() {
		return output;
	}

	/**
	 * Enqueue an DatabaseAction
	 *
	 * @param databaseAction the DatabaseAction to be performed
	 */
	public void enqueue(DatabaseAction databaseAction) {
		try {
			threadPool.execute(new DatabaseOperation(this, databaseAction));
		} catch (RejectedExecutionException ignored) {
		}
	}

	/**
	 * Gets the owner of an Ownable.
	 *
	 * @param ownable the ownable
	 * @return the owner
	 */
	public User getOwner(Ownable ownable) {
		return getDatabaseOwner(ownable);
	}

	/**
	 * Disables database as soon as queue is empty and cache is flushed.
	 */
	public final void disable() {
		disable = true;
	}

	@Override
	public void run() {
		while (!disable) {
			try {
				Thread.sleep(500);
			} catch (InterruptedException ignored) {
			}
		}

		threadPool.shutdown();
		getOutput().debugMessage("Waiting for database actions termination");
		try {
			threadPool.awaitTermination(10, TimeUnit.SECONDS);
		} catch (InterruptedException ignored) {
		}

		close();
	}

	void createTables() throws SQLException {
		Statement stmnt = connection.createStatement();
		for(String query : generateCreateTableQueries()) {
			stmnt.executeUpdate(query);
		}
		stmnt.close();
	}

	/**
	 * Accesses the database and searches for the owner
	 *
	 * @param ownable Ownable of which the owner is searched
	 * @return owner or User.nobody if there is none
	 */
	User getDatabaseOwner(Ownable ownable) {
		try {
			Statement stmnt = connection.createStatement();
			String query;
			query = generateGetOwnerQuery(ownable);
			ResultSet rs = stmnt.executeQuery(query);
			if (rs.next()) {
				User owner;
				owner = User.getInstance(UUID.fromString(rs.getString(PLAYER_ID_COLUMN)));
				stmnt.close();
				return owner;
			} else {
				stmnt.close();
				return User.nobody;
			}
		} catch (SQLException e) {
			getOutput().printException("Couldn't retrieve owner of ownable " + ownable, e);
			return null;
		} catch (IllegalArgumentException e) {
			getOutput().printException("Couldn't retrieve owner of ownable " + ownable, e);
			return null;
		}
	}

	/**
	 * Accesses the database and sets the owner of the Ownable
	 *
	 * @param databaseAction the DatabaseAction to be performed
	 * @return True, if successful
	 */
	boolean setDatabaseOwner(DatabaseAction databaseAction) {
		try {
			Statement stmnt = connection.createStatement();

			if (databaseAction.getActionType() == DatabaseAction.Type.UNOWN) {
				String query;
				try {
					query = generateDeleteOwnerQuery(databaseAction.getOwnable());
				} catch(IllegalArgumentException e) {
					getOutput().printException(e);
					stmnt.close();
					return false;
				}

				stmnt.executeUpdate(query);
				stmnt.close();
				return true;
			} else if (databaseAction.getActionType() == DatabaseAction.Type.OWN) {
				String query;
				try{
					query = generateSetOwnerQuery(databaseAction);
				}catch(IllegalArgumentException e) {
					getOutput().printException(e);
					stmnt.close();
					return false;
				}

				stmnt.executeUpdate(query);
				stmnt.close();
				return true;
			} else {
				// Should never happen
				getOutput().printException(new IllegalArgumentException("Invalid DatabaseActionType"));
				stmnt.close();
				return false;
			}
		} catch (SQLException e) {
			getOutput().printException(e);
			return false;
		}
	}

	/**
	 * Accesses the database and drops all data related to the specified user
	 *
	 * @param user the User
	 */
	void dropDatabaseUserData(User user) {
		if (user.isNobody()) return;
		try {
			Statement stmnt = connection.createStatement();
			for(String query : generateDropUserQueries(user)) {
				stmnt.executeUpdate(query);
			}
			stmnt.close();
		} catch (SQLException e) {
			getOutput().printException("Exception while dropping the owning data for " + user.getName(), e);
		}
	}

	/**
	 * Flush cache data to database.
	 *
	 * @param cacheData the cache data
	 * @return true, if successful
	 */
	boolean flushDatabase(Map<Ownable, User> cacheData) {
		Iterator<Entry<Ownable, User>> iterator = cacheData.entrySet().iterator();
		Entry<Ownable, User> owning;
		while (iterator.hasNext()) {
			owning = iterator.next();
			DatabaseAction action;
			if (owning.getValue().isNobody())
				action = DatabaseAction.newUnownInstance(owning.getKey());
			else
				action = DatabaseAction.newOwnInstance(owning.getKey(), owning.getValue());

			if(!setDatabaseOwner(action)) return false;
			iterator.remove();
		}
		return true;
	}

	/**
	 * Close connection
	 */
	void close() {
		try {
			connection.close();
		} catch (SQLException e) {
			getOutput().printError("Couldn't close database connection", e);
		}
	}

	abstract String[] generateCreateTableQueries();

	abstract String generateGetOwnerQuery(Ownable ownable);

	/**
	 * Generates a sql query to set a new owner for a ownable
	 *
	 * @param databaseAction Guaranteed to have DatabaseActionType OWN and user != User.nobody
	 * @return the query
	 */
	abstract String generateSetOwnerQuery(DatabaseAction databaseAction);

	abstract String generateDeleteOwnerQuery(Ownable ownable);

	abstract String[] generateDropUserQueries(User user);
}

class DatabaseOperation implements Runnable {

	private final DatabaseAction action;
	private final Database database;

	DatabaseOperation(Database database, DatabaseAction action) {
		this.action = action;
		this.database = database;
	}

	@Override
	public void run() {
		if (action.getActionType() == DatabaseAction.Type.DROP)
			database.dropDatabaseUserData(action.getUser());
		else if (!database.setDatabaseOwner(action))
			database.getOutput().printException("An action has not been performed", new ActionNotPerformedException(action));
	}
}
