package de.pheasn.blockown.database;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import de.pheasn.blockown.Ownable;
import de.pheasn.blockown.User;
import org.bukkit.Bukkit;

import java.io.*;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

class CacheAccessor {

	/**
	 * Estimated amount of blocks one player will place
	 */
	private static final int BLOCKS_PER_PLAYER = 1024;
	private static final int DEFAULT_START_CAPACITY = 65536;
	private static final int CACHE_TIME = 15;

	private final File dumpFile;
	private final CachedDatabase db;

	private final LoadingCache<Ownable, User> cache;
	private Map<Ownable, User> unsubmitted;

	CacheAccessor(CachedDatabase database, File pluginFolder) {
		this.db = database;
		this.cache = CacheBuilder.newBuilder().maximumSize(calculateCacheCapacity()).expireAfterAccess(CACHE_TIME, TimeUnit.MINUTES).build(new CacheLoader<Ownable, User>() {
			@Override
			public User load(Ownable ownable) throws Exception {
				return db.getDatabaseOwner(ownable);
			}
		});

		dumpFile = new File(pluginFolder.getAbsoluteFile(), "dumpFile.dat");
		if (dumpFile.exists()) {
			unsubmitted = restoreDumpedCache();
			cache.putAll(unsubmitted);
		} else {
			unsubmitted = initializeCache();
		}
	}

	/**
	 * Creates a new HashMap with an appropriate capacity
	 *
	 * @return the HashMap
	 */
	private Map<Ownable, User> initializeCache() {
		return new ConcurrentHashMap<>(calculateCacheCapacity());
	}

	private int calculateCacheCapacity() {
		int initialCapacity;
		try {
			initialCapacity = Bukkit.getOfflinePlayers().length * BLOCKS_PER_PLAYER;
		} catch (NullPointerException e) {
			// Neccessary for testing
			initialCapacity = 0;
		}
		if (initialCapacity == 0) initialCapacity = DEFAULT_START_CAPACITY;
		return initialCapacity;
	}

	/**
	 * Restores cacheAccessor from dump file and deletes file afterwards. If unsuccessful, returns empty cacheAccessor
	 *
	 * @return the restored cacheAccessor
	 */
	@SuppressWarnings("unchecked")
	private Map<Ownable, User> restoreDumpedCache() {
		Map<Ownable, User> result = null;
		try {
			ObjectInputStream ois = new ObjectInputStream(new FileInputStream(dumpFile));
			Object o = ois.readObject();
			ois.close();
			if (o instanceof Map<?, ?>) {
				assert ((Map<?, ?>) o).keySet().iterator().next() instanceof Ownable;
				assert ((Map<?, ?>) o).values().iterator().next() instanceof User;
				result = (Map<Ownable, User>) o;
			}
			dumpFile.delete();
		} catch (ClassNotFoundException | IOException e) {
			db.getOutput().printException("Error restoring dumped cacheAccessor.", e);
			if (!dumpFile.delete()) {
				db.getOutput().printError("Couldn't delete corrupted cache dump, please delete " + dumpFile.getAbsolutePath() + " manually", e);
			}
		}
		return (result == null) ? initializeCache() : result;
	}

	User getOwner(Ownable ownable) {
		try {
			return cache.get(ownable);
		} catch (ExecutionException e) {
			db.getOutput().printException("ExecutionException in CacheAccessor", e);
			return User.nobody;
		}
	}

	boolean doAction(DatabaseAction databaseAction) {
		switch (databaseAction.getActionType()) {
		case UNOWN:
			cache.put(databaseAction.getOwnable(), User.nobody);
			unsubmitted.put(databaseAction.getOwnable(), User.nobody);
			return true;
		case OWN:
			cache.put(databaseAction.getOwnable(), databaseAction.getUser());
			unsubmitted.put(databaseAction.getOwnable(), databaseAction.getUser());
			return true;
		case DROP:
			dropUserData(databaseAction.getUser());
			return true;
		default:
			db.getOutput().printException(new IllegalArgumentException("Invalid DatabaseActionType"));
			return false;
		}
	}

	/**
	 * Flush all cacheAccessor data to the db
	 */
	synchronized void flush() {
		if (!db.flushDatabase(unsubmitted)) {
			try {
				dumpFile.createNewFile();
				ObjectOutputStream oos = new ObjectOutputStream(new FileOutputStream(dumpFile, false));
				oos.writeObject(unsubmitted);
				oos.close();
			} catch (IOException e) {
				db.getOutput().printException("Exception while flushing cache", e);
			}
			db.getOutput().printConsole("The cache has been dumped to the hard drive");
		}
	}

	private void dropUserData(User user) {
		if (user == null) return;
		Iterator<Entry<Ownable, User>> iterator = cache.asMap().entrySet().iterator();
		Entry<Ownable, User> entry;
		while (iterator.hasNext()) {
			entry = iterator.next();
			if (user.equals(entry.getValue())) {
				cache.put(entry.getKey(), User.nobody);
			}
		}
		iterator = unsubmitted.entrySet().iterator();
		while (iterator.hasNext()) {
			entry = iterator.next();
			if (user.equals(entry.getValue())) {
				entry.setValue(User.nobody);
			}
		}
		db.dropDatabaseUserData(user);
	}

}
