package de.pheasn.blockown.command;

import de.pheasn.blockown.BlockOwn;
import de.pheasn.blockown.Message;
import de.pheasn.blockown.database.Database;
import de.pheasn.blockown.importer.Importer;
import de.pheasn.blockown.importer.ImporterException;
import org.bukkit.command.Command;
import org.bukkit.command.CommandSender;
import org.bukkit.entity.Player;

import java.io.File;
import java.lang.reflect.InvocationTargetException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;

public class CE_Import extends CommandExecutor {

	private final ClassLoader externalImportClassLoader;
	private final boolean folderAccess;

	public CE_Import(BlockOwn plugin) {
		super(plugin);
		URL url = null;
		try {
			url = plugin.getImporterFolder().toURI().toURL();
		} catch (MalformedURLException e) {
			plugin.getOutput().printException("Error accessing Importer folder, import command unavailable", e);
			folderAccess = false;
			externalImportClassLoader = null;
			return;
		}
		if (url != null) {
			externalImportClassLoader = URLClassLoader.newInstance(
					new URL[]{url}, plugin.getClass().getClassLoader());
		} else {
			externalImportClassLoader = null;
		}
		folderAccess = true;
	}

	@Override
	public boolean onCommand(final CommandSender sender, Command cmd, String cmdName, String[] args) {
		if(!folderAccess) {
			plugin.getOutput().sendMessage(sender, Message.COMMAND_IMPORT_UNAVAILABLE);
			return false;
		}

		if (sender instanceof Player) {
			plugin.getOutput().sendMessage(sender, Message.COMMAND_CONSOLE_ONLY);
			return false;
		}

		if (args.length != 1) return false;
		Class<? extends Importer> importerClass;
		try {
			importerClass = findClass(args[0]);
		} catch (ClassNotFoundException e) {
			plugin.getOutput().printException("No matching importer found", e);
			return false;
		}
		try {
			Importer importer = importerClass.getConstructor(Database.class, File.class, Runnable.class).newInstance(
					plugin.getOwningDatabase(),
					new File("plugins"),
					new Runnable() {
						@Override
						public void run() {
							plugin.getOutput().sendMessage(sender, Message.COMMAND_IMPORT_SUCCESS);
						}

					});
			Thread importThread = new Thread(importer, "Import Thread");
			plugin.getOutput().sendMessage(sender, Message.COMMAND_IMPORT_START);
			importThread.start();
			return true;
		} catch (InstantiationException | IllegalAccessException | InvocationTargetException | NoSuchMethodException | SecurityException e) {
			plugin.getOutput().printException("There probably was an error in the requested importer class", e);
			return false;
		} catch (ImporterException e) {
			plugin.getOutput().printException(e.getMessage(), e);
			return false;
		}
	}

	@Override
	protected boolean performCommand(Player player, Command cmd, String cmdName, String[] args) {
		return false;
	}

	@SuppressWarnings("unchecked")
	private Class<? extends Importer> findClass(String name) throws ClassNotFoundException {
		Class<? extends Importer> result = null;
		try {
			Class<?> found = Class.forName("de.pheasn.blockown.importer." + name);
			if (Importer.class.isAssignableFrom(found) && !found.isInterface()) {
				result = (Class<? extends Importer>) found;
			}
		} catch (ClassNotFoundException e) {
			if (externalImportClassLoader != null) {
				Class<?> found = externalImportClassLoader.loadClass(name);
				if (Importer.class.isAssignableFrom(found)
						&& !found.isInterface()) {
					result = (Class<? extends Importer>) found;
				}
			}
		}
		if (result == null) throw new ClassNotFoundException();
		return result;
	}

}
