package de.pheasn.blockown;

import org.bukkit.Bukkit;
import org.bukkit.OfflinePlayer;
import org.bukkit.World;
import org.bukkit.entity.Player;

import java.io.Serializable;
import java.lang.ref.WeakReference;
import java.util.Map;
import java.util.UUID;
import java.util.WeakHashMap;

public class User implements Serializable {

    private static final long serialVersionUID = -9170381319637300740L;
    public static final User nobody = new User(null, null);
    private static final Map<UUID, User> users = new WeakHashMap<UUID, User>(128);
    private static BlockOwn plugin = null;

    private final UUID id;
    private transient WeakReference<OfflinePlayer> playerRef = new WeakReference<>(null);

    public static User getInstance(UUID id) {
        return getInstance(id, null);
    }

    public static User getInstance(UUID id, OfflinePlayer player) {
        if (id == null) throw new NullPointerException("UUID can't be null");
        synchronized (users) {
            User user = users.get(id);
            if (user == null) {
                user = new User(id, player);
                users.put(id, user);
            } else {
                user.playerRef = new WeakReference<>(player);
            }
            return user;
        }
    }

    private User(UUID id, OfflinePlayer player) {
        this.id = id;
        this.playerRef = new WeakReference<>(player);
    }

    static synchronized void initialize(BlockOwn plugin) {
        if(plugin == null) throw new NullPointerException("Can't initialize with plugin = null");
        User.plugin = plugin;
    }

    public UUID getId() {
        return id;
    }

    public boolean isOnline() {
        OfflinePlayer o = getOfflinePlayer();
        return (o != null) && o.isOnline();
    }

    public boolean isOp() {
        OfflinePlayer o = getOfflinePlayer();
        return (o != null) && o.isOp();
    }

    /**
     * Gets the last known name of the Player represented by this User
     *
     * @return the name
     */
    public String getName() {
        OfflinePlayer o = getOfflinePlayer();
        return (o == null) ? null : o.getName();
    }

    public OfflinePlayer getOfflinePlayer() {
		if (isNobody()) return null;
		if (playerRef == null) playerRef = new WeakReference<>(null);
		OfflinePlayer offlinePlayer = playerRef.get();
		if (offlinePlayer == null) {
			try {
				offlinePlayer = getPlugin().getServer().getOfflinePlayer(id);
				playerRef = new WeakReference<>(offlinePlayer);
			} catch (NullPointerException e) {
				return null;
			}
		}
        return offlinePlayer;
    }

    public World getWorld() {
        OfflinePlayer offlinePlayer = getOfflinePlayer();
        if (offlinePlayer == null)
            return null;
        Player player = offlinePlayer.getPlayer();
        if (player == null)
            return null;
        return player.getWorld();
    }

    public boolean isNobody() {
        return id == null;
    }

    private BlockOwn getPlugin() {
        if (plugin == null)
            initialize((BlockOwn) Bukkit.getPluginManager().getPlugin("BlockOwn"));
        return plugin;
    }

    public boolean hasAccess(Ownable ownable) throws InvalidWorldException {
        User owner = getPlugin().getOwningDatabase().getOwner(ownable);
        return owner.isNobody() || hasAccess(ownable.getMaterial(), owner);
    }

    public boolean hasAccess(Material material, User owner) {
        return getPlugin().getProtection().hasAccess(owner, material, this);
    }

    public boolean hasProtected(Material material) {
        return getPlugin().getProtection().isProtected(this, material);
    }

    public boolean hasLocked(Material material) {
        return getPlugin().getProtection().isLocked(this, material);
    }

    public boolean hasFriend(User user) {
        return getPlugin().getProtection().isFriend(this, user);
    }

    public boolean isIgnoring() {
        return getPlugin().isIgnoring(this);
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 5;
        result = prime * result + ((id == null) ? 0 : id.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (!(obj instanceof User))
            return false;
        User other = (User) obj;
        if (id == null) {
            if (other.id != null)
                return false;
        } else if (!id.equals(other.id))
            return false;
        return true;
    }

    @Override
    public String toString() {
        String name = getName();
        return name == null ? (id == null ? "nobody" : id.toString()) : name;
    }
}
