package de.pheasn.blockown.database;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Map;
import java.util.UUID;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentLinkedQueue;

import de.pheasn.blockown.Output;
import de.pheasn.blockown.Ownable;
import de.pheasn.blockown.OwnedBlock;
import de.pheasn.blockown.OwnedEntity;
import de.pheasn.blockown.User;

public abstract class Database implements Runnable {

	// Column names
	protected static final String PLAYER_ID_COLUMN = "player_id";
	protected static final String WORLD_COLUMN = "world";
	protected static final String ENTITY_ID_COLUMN = "entity_id";
	protected static final String X_COLUMN = "x";
	protected static final String Y_COLUMN = "y";
	protected static final String Z_COLUMN = "z";

	// Table names
	protected static final String BLOCK_TABLE = "block_table";
	protected static final String ENTITY_TABLE = "entity_table";

	protected final ConcurrentLinkedQueue<DatabaseAction> queue;
	protected volatile boolean disable = false;
	protected final Object LOCK = new Object();

	protected final Output output;

	protected Connection connection;

	protected Database(Output output) {
		this.output = output;
		this.queue = new ConcurrentLinkedQueue<DatabaseAction>();
	}

	protected Output getOutput() {
		return output;
	}

	/**
	 * Enqueues an DatabaseAction
	 *
	 * @param databaseAction
	 * @return the enqueue thread
	 */
	public final Thread enqueue(final DatabaseAction databaseAction) {
		Thread t = new Thread(new Runnable() {
			@Override
			public void run() {
				syncedEnqueue(databaseAction);
			}
		}, "DatabaseEnqueueThread");
		t.start();
		return t;
	}

	protected final void syncedEnqueue(DatabaseAction databaseAction) {
		queue.add(databaseAction);
		synchronized (LOCK) {
			LOCK.notifyAll();
		}
	}

	/**
	 * Gets the owner of an Ownable.
	 *
	 * @param ownable
	 *            the ownable
	 * @return the owner
	 */
	public User getOwner(Ownable ownable) {
		return getDatabaseOwner(ownable);
	}

	/**
	 * Disables database as soon as queue is empty and cache is flushed.
	 */
	public final void disable() {
		synchronized (LOCK) {
			disable = true;
			LOCK.notifyAll();
		}
	}

	@Override
	public void run() {
		DatabaseAction databaseAction;
		while (!disable || !queue.isEmpty()) {
			if (!queue.isEmpty()) {
				databaseAction = queue.remove();
				if (databaseAction.getActionType() == DatabaseActionType.DROP) {
					dropDatabaseUserData(databaseAction.getUser());
				} else {
					if (!setDatabaseOwner(databaseAction)) {
						getOutput().printException("An action has not been performed", new ActionNotPerformedException(databaseAction));
					}
				}
			} else {
				try {
					synchronized (LOCK) {
						while (!disable && queue.isEmpty()) {
							LOCK.wait();
						}
					}
				} catch (InterruptedException e) {
					getOutput().printException("Database thread has been interrupted.", e);
				}
			}
		}
		close();
	}

	/**
	 * Accesses the database and searches for the owner
	 * 
	 * @param ownable
	 *            Ownable of which the owner is searched
	 * @return owner or User.nobody if there is none
	 */
	protected User getDatabaseOwner(Ownable ownable) {
		try {
			Statement stmnt = connection.createStatement();
			String query = null;
			if (ownable instanceof OwnedBlock) {
				query = generateGetOwnerQuery((OwnedBlock) ownable);
			} else if (ownable instanceof OwnedEntity) {
				query = generateGetOwnerQuery((OwnedEntity) ownable);
			} else {
				getOutput().printException(new UnknownOwnableException());
				stmnt.close();
				return null;
			}
			ResultSet rs = stmnt.executeQuery(query);
			if (rs.next()) {
				User owner;
				owner = new User(UUID.fromString(rs.getString(PLAYER_ID_COLUMN)));
				stmnt.close();
				return owner;
			} else {
				stmnt.close();
				return User.nobody;
			}
		} catch (SQLException e) {
			getOutput().printException(e);
			return null;
		}
	}

	/**
	 * Accesses the database and sets the owner of the Ownable
	 *
	 * @param databaseAction
	 *            the DatabaseAction to be performed
	 * @return True, if successful
	 */
	protected boolean setDatabaseOwner(DatabaseAction databaseAction) {
		try {
			Statement stmnt = connection.createStatement();

			if (databaseAction.getActionType() == DatabaseActionType.UNOWN || databaseAction.getUser() == null || databaseAction.getUser().isNobody()) {
				String query = null;
				if (databaseAction.getOwnable() instanceof OwnedBlock) {
					query = generateDeleteOwnerQuery((OwnedBlock) databaseAction.getOwnable());
				} else if (databaseAction.getOwnable() instanceof OwnedEntity) {
					query = generateDeleteOwnerQuery((OwnedEntity) databaseAction.getOwnable());
				} else {
					getOutput().printException(new UnknownOwnableException());
					stmnt.close();
					return false;
				}
				stmnt.executeUpdate(query);
				stmnt.close();
				return true;
			} else if (databaseAction.getActionType() == DatabaseActionType.OWN) {
				String query = null;
				if (databaseAction.getOwnable() instanceof OwnedBlock) {
					query = generateSetBlockOwnerQuery(databaseAction);
				} else if (databaseAction.getOwnable() instanceof OwnedEntity) {
					query = generateSetEntityOwnerQuery(databaseAction);
				} else {
					getOutput().printException(new UnknownOwnableException());
					stmnt.close();
					return false;
				}
				stmnt.executeUpdate(query);
				stmnt.close();
				return true;
			} else {
				getOutput().printException(new IllegalArgumentException());
				stmnt.close();
				return false;
			}
		} catch (SQLException e) {
			getOutput().printException(e);
			return false;
		}
	}

	/**
	 * Accesses the database and drops all data related to the specified user
	 *
	 * @param user
	 *            the User
	 */
	protected void dropDatabaseUserData(User user) {
		if (user.isNobody()) return;
		try {
			Statement stmnt = connection.createStatement();
			String query = null;
			query = generateDropUserBlocksQuery(user);
			stmnt.executeUpdate(query);
			query = generateDropUserEntitiesQuery(user);
			stmnt.executeUpdate(query);
			stmnt.close();
		} catch (SQLException e) {
			getOutput().printException(e);
			return;
		}
	}

	/**
	 * Flush cache data to database.
	 *
	 * @param cacheData
	 *            the cache data
	 * @return true, if successful
	 */
	protected boolean flushDatabase(Map<Ownable, User> cacheData) {
		for (Entry<Ownable, User> owning : cacheData.entrySet()) {
			if (owning.getValue().isNobody()) {
				if (!setDatabaseOwner(new DatabaseAction(DatabaseActionType.UNOWN, owning.getKey(), null))) {
					return false;
				}
			} else {
				if (!setDatabaseOwner(new DatabaseAction(DatabaseActionType.OWN, owning.getKey(), owning.getValue()))) {
					return false;
				}
			}
		}
		return true;
	}

	/**
	 * Close connection
	 */
	protected void close() {
		try {
			connection.close();
		} catch (SQLException e) {
			getOutput().printException("Couldn't close database connection", e);
		}
	}

	protected abstract String generateCreateEntityTableQuery();

	protected abstract String generateCreateBlockTableQuery();

	protected abstract String generateGetOwnerQuery(OwnedBlock block);

	protected abstract String generateGetOwnerQuery(OwnedEntity entity);

	/**
	 * Generates a sql query to set a new owner for a block
	 * 
	 * @param databaseAction
	 *            Guaranteed to have DatabaseActionType OWN and user != User.nobody
	 * @return the query
	 */
	protected abstract String generateSetBlockOwnerQuery(DatabaseAction databaseAction);

	/**
	 * Generates a sql query to set a new owner for a entity
	 * 
	 * @param databaseAction
	 *            Guaranteed to have DatabaseActionType OWN and user != User.nobody
	 * @return the query
	 */
	protected abstract String generateSetEntityOwnerQuery(DatabaseAction databaseAction);

	protected abstract String generateDeleteOwnerQuery(OwnedEntity entity);

	protected abstract String generateDeleteOwnerQuery(OwnedBlock block);

	protected abstract String generateDropUserBlocksQuery(User user);

	protected abstract String generateDropUserEntitiesQuery(User user);
}
