package de.pheasn.blockown;

import java.io.File;
import java.io.IOException;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.bukkit.Bukkit;
import org.bukkit.World;
import org.bukkit.entity.Animals;
import org.bukkit.entity.LivingEntity;
import org.bukkit.plugin.java.JavaPlugin;
import org.mcstats.Metrics;

import de.pheasn.blockown.command.CE_AddFriend;
import de.pheasn.blockown.command.CE_Ignore;
import de.pheasn.blockown.command.CE_Import;
import de.pheasn.blockown.command.CE_List;
import de.pheasn.blockown.command.CE_Lock;
import de.pheasn.blockown.command.CE_Own;
import de.pheasn.blockown.command.CE_Protect;
import de.pheasn.blockown.command.CE_RemFriend;
import de.pheasn.blockown.command.CE_ShowOwner;
import de.pheasn.blockown.command.CE_Unlock;
import de.pheasn.blockown.command.CE_Unown;
import de.pheasn.blockown.command.CE_UnownPlayer;
import de.pheasn.blockown.command.CE_Unprotect;
import de.pheasn.blockown.command.CE_UnprotectPlayer;
import de.pheasn.blockown.database.Database;
import de.pheasn.blockown.database.MySqlDatabase;
import de.pheasn.blockown.database.SqliteDatabase;
import de.pheasn.blockown.event.L_BlockBreak;
import de.pheasn.blockown.event.L_BlockPlace;
import de.pheasn.blockown.event.L_EnvironmentDamage;
import de.pheasn.blockown.event.L_PlayerInteractEvent;
import de.pheasn.blockown.protection.Protection;
import de.pheasn.pluginupdater.PluginVersion;
import de.pheasn.pluginupdater.ReleaseChannel;
import de.pheasn.pluginupdater.Updatable;
import de.pheasn.pluginupdater.Updater;
import de.pheasn.pluginupdater.UpdaterKey;

public class BlockOwn extends JavaPlugin implements Updatable {

	private static final String DATA_FOLDER_PATH = "plugins" + File.separator + "BlockOwn";
	private static final int PLUGIN_ID = 62749;
	private static final ReleaseChannel releaseChannel = ReleaseChannel.BETA;

	private Output output;

	private Database database;
	private Thread databaseThread;

	private Protection protection;
	private Thread protectionThread;

	private Updater updater;

	private Set<String> enabledWorlds;
	private Set<Material> ownEnabledMaterials;
	private Set<Material> ownDisabledMaterials;

	private File messageFile;
	private File configFile;
	private File protectionFile;
	private File importerFolder;

	private Set<User> ignoringUsers;
	private Map<User, WaitType> waitingUsers;

	@Override
	public void onDisable() {
		database.disable();
		protection.disable();
		while (databaseThread.isAlive() || protectionThread.isAlive()) {
			try {
				Thread.sleep(1000);
			} catch (InterruptedException e) {
			}
		}

		saveConfig();
		super.onDisable();
	}

	@Override
	public void onEnable() {
		super.onEnable();
		Setting.setConfig(getConfig());
		output = Setting.DEBUG_MODE.get() ? new DebugOutput() : new DefaultOutput();
		Setting.setOutput(output);
		User.initialize(this);

		messageFile = new File(getDataFolder(), "messages.yml");
		configFile = new File(getDataFolder(), "config.yml");
		protectionFile = new File(getDataFolder(), "protection.dat");
		importerFolder = new File("./plugins/BlockOwn/Importer");

		updateDeprecatedSettings();

		if (!createDataStructure()) {
			getOutput().printException("Error creating data structure.");
			Bukkit.getPluginManager().disablePlugin(this);
			return;
		}

		if (!initializeDatabase()) {
			getOutput().printException("Couldn't connect to database.");
			Bukkit.getPluginManager().disablePlugin(this);
			return;
		}

		if (!initializeProtection()) {
			getOutput().printException("Couldn't initialize Protection.");
			Bukkit.getPluginManager().disablePlugin(this);
			return;
		}

		enabledWorlds = getEnabledWorlds();
		ownEnabledMaterials = getOwnEnabledMaterials();
		ownDisabledMaterials = getOwnDisabledMaterials();
		ignoringUsers = new HashSet<User>(getServer().getOfflinePlayers().length * 2);
		waitingUsers = new HashMap<User, WaitType>(getServer().getOfflinePlayers().length * 2);

		registerListeners();
		registerCommands();

		updater = new Updater(this, PLUGIN_ID, 30 * 60 * 1000, getFile(), Setting.UPDATER_API_KEY.get(), Setting.UPDATER_RELEASE_CHANNEL.getReleaseChannel());
		updater.start();

		initializeMetrics();

		Setting.VERSION.set(this.getDescription(UpdaterKey.VERSION) + "-" + getReleaseChannel());
	}

	@SuppressWarnings("deprecation")
	private void updateDeprecatedSettings() {
		PluginVersion oldVersion = new PluginVersion(Setting.VERSION.get(), getReleaseChannel());
		for(Setting.OLD oldSetting : Setting.OLD.values()) {
			PluginVersion deprecatingVersion = new PluginVersion(oldSetting.getDeprecatingVersion(), getReleaseChannel());
			if(deprecatingVersion.compareTo(oldVersion) > 0) {
				oldSetting.transferValue();
				oldSetting.getSetting().set(null);
			}
		}
	}

	private void initializeMetrics() {
		try {
			Metrics metrics = new Metrics(this);
			metrics.start();
		} catch (IOException e) {
			// Failed to submit the stats
		}
	}

	private Set<String> getEnabledWorlds() {
		HashSet<String> result = new HashSet<String>(20);
		List<String> disabled = Setting.DISABLED_WORLDS.get();
		List<String> enabled = Setting.ENABLED_WORLDS.get();

		if (disabled.isEmpty() && enabled.isEmpty()) {
			for (World world : getServer().getWorlds()) {
				result.add(world.getName());
			}
			return result;
		}

		Set<String> disabledLower = new HashSet<String>(disabled.size());
		for (String s : disabled)
			disabledLower.add(s.toLowerCase());
		Set<String> enabledLower = new HashSet<String>(enabled.size());
		for (String s : enabled)
			enabledLower.add(s.toLowerCase());

		for (World world : getServer().getWorlds()) {
			if (disabledLower.contains(world.getName().toLowerCase())) continue;
			if (enabledLower.isEmpty() || enabledLower.contains(world.getName().toLowerCase())) {
				result.add(world.getName());
			}
		}
		return result;
	}

	private Set<Material> getOwnEnabledMaterials() {
		List<String> enabled = Setting.DATABASE_ALLOWED_MATERIALS.get();
		HashSet<Material> result = new HashSet<Material>(enabled.size(), 1);
		Material m;
		for (String materialName : enabled) {
			try {
				m = Material.parseMaterial(materialName);
				result.add(m);
			} catch (IllegalArgumentException e) {
			}
		}
		return result;
	}

	private Set<Material> getOwnDisabledMaterials() {
		List<String> disabled = Setting.DATABASE_DISALLOWED_MATERIALS.get();
		HashSet<Material> result = new HashSet<Material>(disabled.size(), 1);
		Material m;
		for (String materialName : disabled) {
			try {
				m = Material.parseMaterial(materialName);
				result.add(m);
			} catch (IllegalArgumentException e) {
			}
		}
		return result;
	}

	public boolean isOwnEnabled(Material material) {
		if (ownDisabledMaterials.contains(material)) return false;
		if (ownEnabledMaterials.contains(material)) return true;
		if (material.isBlock()) return ownEnabledMaterials.isEmpty();
		if (Animals.class.isAssignableFrom(material.getEntityType().getEntityClass())) {
			return Setting.DATABASE_ALLOW_ANIMALS.get();
		}
		if (!LivingEntity.class.isAssignableFrom(material.getEntityType().getEntityClass())) {
			return Setting.DATABASE_ALLOW_ENTITIES.get();
		}
		return false;
	}

	public boolean isIgnoring(User user) {
		return ignoringUsers.contains(user);
	}

	public void setIgnoring(User user, boolean enable) {
		if (enable) {
			ignoringUsers.add(user);
		} else {
			ignoringUsers.remove(user);
		}
	}

	public void addWaiting(User user, WaitType waitType) {
		waitingUsers.put(user, waitType);
	}

	public boolean isWaiting(User user, WaitType waitType) {
		return waitingUsers.get(user) == waitType;
	}

	public WaitType getWaiting(User user) {
		return waitingUsers.get(user);
	}

	public void removeWaiting(User user) {
		waitingUsers.remove(user);
	}

	private boolean createDataStructure() {
		File dataFolder = getDataFolder();
		if (!dataFolder.exists()) {
			if (!dataFolder.mkdir()) return false;
		}
		getConfig().options().copyDefaults(true);
		if (!configFile.exists()) saveDefaultConfig();
		if (!Message.loadMessages(this)) return false;
		if (!importerFolder.exists()) {
			if (!importerFolder.mkdir()) return false;
		}
		return true;
	}

	private boolean initializeDatabase() {
		if (Setting.DATABASE_MYSQL_ENABLE.get()) {
			try {
				if (Setting.DATABASE_MYSQL_PASSWORD.get().equals("")) {
					database = new MySqlDatabase(getOutput(),
							getDataFolder(),
							Setting.DATABASE_MYSQL_HOST.get(),
							Setting.DATABASE_MYSQL_PORT.get(),
							Setting.DATABASE_MYSQL_DATABASE.get(),
							Setting.DATABASE_MYSQL_USER.get());
				} else {
					database = new MySqlDatabase(getOutput(),
							getDataFolder(),
							Setting.DATABASE_MYSQL_HOST.get(),
							Setting.DATABASE_MYSQL_PORT.get(),
							Setting.DATABASE_MYSQL_DATABASE.get(),
							Setting.DATABASE_MYSQL_USER.get(),
							Setting.DATABASE_MYSQL_PASSWORD.get());
				}
				databaseThread = new Thread(database, "DatabaseThread");
				databaseThread.start();
				return true;
			} catch (ClassNotFoundException e) {
				getOutput().printException("MySQL driver class not found");
				return false;
			} catch (SQLException e) {
				getOutput().printException("Exception during MySQL initialization", e);
				return false;
			}
		} else {
			try {
				database = new SqliteDatabase(getOutput(), getDataFolder());
				databaseThread = new Thread(database, "DatabaseThread");
				databaseThread.start();
				return true;
			} catch (ClassNotFoundException e) {
				getOutput().printException("SQLite driver class not found");
				return false;
			} catch (SQLException e) {
				getOutput().printException("Exception during SQLite initialization", e);
				return false;
			}
		}
	}

	private boolean initializeProtection() {
		try {
			protection = new Protection(getOutput(), getProtectionFile());
			protectionThread = new Thread(protection, "ProtectionThread");
			protectionThread.start();
			return true;
		} catch (IOException e) {
			getOutput().printException("Exception while loading protections file", e);
			return false;
		}
	}

	private void registerListeners() {
		getServer().getPluginManager().registerEvents(new L_BlockPlace(this), this);
		getServer().getPluginManager().registerEvents(new L_EnvironmentDamage(this), this);
		getServer().getPluginManager().registerEvents(new L_PlayerInteractEvent(this), this);
		getServer().getPluginManager().registerEvents(new L_BlockBreak(this), this);
	}

	private void registerCommands() {
		getCommand(Command.IGNORE.toString()).setExecutor(new CE_Ignore(this));
		getCommand(Command.IGNORE.toString()).setUsage("/<command> [opt.] on/off");
		getCommand(Command.IMPORT.toString()).setExecutor(new CE_Import(this));
		getCommand(Command.IMPORT.toString()).setUsage("/<command> <importer-name>");
		getCommand(Command.SHOWOWNER.toString()).setExecutor(new CE_ShowOwner(this));
		getCommand(Command.SHOWOWNER.toString()).setUsage("/<command> [opt.] e");
		getCommand(Command.ADDFRIEND.toString()).setExecutor(new CE_AddFriend(this));
		getCommand(Command.ADDFRIEND.toString()).setUsage("/<command> <player>");
		getCommand(Command.REMFRIEND.toString()).setExecutor(new CE_RemFriend(this));
		getCommand(Command.REMFRIEND.toString()).setUsage("/<command> <player>");
		getCommand(Command.LOCK.toString()).setExecutor(new CE_Lock(this));
		getCommand(Command.LOCK.toString()).setUsage("/<command> [opt.] e / <material-name>");
		getCommand(Command.UNLOCK.toString()).setExecutor(new CE_Unlock(this));
		getCommand(Command.UNLOCK.toString()).setUsage("/<command> [opt.] e / <material-name>");
		getCommand(Command.PROTECT.toString()).setExecutor(new CE_Protect(this));
		getCommand(Command.PROTECT.toString()).setUsage("/<command> [opt.] e / <material-name>");
		getCommand(Command.UNPROTECT.toString()).setExecutor(new CE_Unprotect(this));
		getCommand(Command.UNPROTECT.toString()).setUsage("/<command> [opt.] e / <material-name>");
		getCommand(Command.OWN.toString()).setExecutor(new CE_Own(this));
		getCommand(Command.OWN.toString()).setUsage("/<command> [opt.] e");
		getCommand(Command.UNOWN.toString()).setExecutor(new CE_Unown(this));
		getCommand(Command.UNOWN.toString()).setUsage("/<command> [opt.] e");
		getCommand(Command.LIST.toString()).setExecutor(new CE_List(this));
		getCommand(Command.LIST.toString()).setUsage("/<command> protected/locked/friends");
		getCommand(Command.UNPROTECTPLAYER.toString()).setExecutor(new CE_UnprotectPlayer(this));
		getCommand(Command.UNPROTECTPLAYER.toString()).setUsage("/<command> <player>");
		getCommand(Command.UNOWNPLAYER.toString()).setExecutor(new CE_UnownPlayer(this));
		getCommand(Command.UNOWNPLAYER.toString()).setUsage("/<command> <player>");
	}

	public Database getOwningDatabase() {
		return database;
	}

	public Protection getProtection() {
		return protection;
	}

	public boolean isEnabledInWorld(World world) {
		return enabledWorlds.contains(world.getName());
	}

	@Override
	public void updateIOException(IOException e) {
		getOutput().printException("Error while updating", e);
	}

	@Override
	public void updateInterruptedException(InterruptedException e) {
		getOutput().printException("Error while updating", e);
	}

	@Override
	public String getDescription(UpdaterKey key) {
		switch (key) {
		case NAME:
			return getDescription().getName();
		case MAIN_AUTHOR:
			return getDescription().getAuthors().get(0);
		case VERSION:
			return getDescription().getVersion();
		default:
			return null;
		}
	}

	@Override
	public void updateNotifyDownloading() {
		getOutput().printConsole(Message.UPDATER_DOWNLOADING);
	}

	@Override
	public void updateNotifyRestarting() {
		getOutput().printConsole(Message.UPDATER_RESTART);
	}

	@Override
	public void updateNotifyReadUpdatedProjectPage() {
		getOutput().printConsole(Message.UPDATER_BIG_UPDATE);
	}

	@Override
	public ReleaseChannel getReleaseChannel() {
		return releaseChannel;
	}

	public static String getDataFolderPath() {
		return DATA_FOLDER_PATH;
	}

	public Output getOutput() {
		return output;
	}

	public File getMessageFile() {
		return messageFile;
	}

	public File getProtectionFile() {
		return protectionFile;
	}

	public File getImporterFolder() {
		return importerFolder;
	}
}
