package de.pheasn.blockown.protection;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedQueue;

import org.bukkit.Bukkit;
import org.bukkit.entity.Animals;
import org.bukkit.entity.LivingEntity;

import de.pheasn.blockown.BlockOwn;
import de.pheasn.blockown.Material;
import de.pheasn.blockown.Setting;
import de.pheasn.blockown.User;

public class Protection implements Runnable {

	private final Object LOCK = new Object();
	private volatile boolean disable = false;
	private final BlockOwn plugin;

	private final int initialUserCapacity;
	private final int initialMaterialCapacity = 50;
	private final Map<User, Set<Material>> protections;
	private final Map<User, Set<Material>> locks;
	private final Map<User, Set<User>> friends;

	private final ConcurrentLinkedQueue<ProtectAction> queue;

	private final Set<Material> allowedMaterials;
	private final Set<Material> disallowedMaterials;
	private final Set<Material> autoProtectedMaterials;

	@SuppressWarnings("unchecked")
	public Protection(BlockOwn plugin) throws IOException {
		this.plugin = plugin;
		queue = new ConcurrentLinkedQueue<ProtectAction>();

		allowedMaterials = getAllowedMaterials();
		disallowedMaterials = getDisallowedMaterials();
		autoProtectedMaterials = getAutoProtectedMaterials();

		int initialCapacity;
		try {
			initialCapacity = Bukkit.getOfflinePlayers().length * 2;
		} catch (NullPointerException e) {
			// Neccessary for tests
			initialCapacity = 0;
		}
		if (initialCapacity == 0) initialCapacity = 50;
		initialUserCapacity = initialCapacity;
		if (plugin.getProtectionFile().exists()) {
			Map<User, Set<Material>> loadedProtections = null;
			Map<User, Set<Material>> loadedLocks = null;
			Map<User, Set<User>> loadedFriends = null;
			ObjectInputStream ois = new ObjectInputStream(new FileInputStream(plugin.getProtectionFile()));
			try {
				Object o = ois.readObject();
				assert o instanceof HashMap<?, ?>;
				loadedProtections = (Map<User, Set<Material>>) o;

				o = ois.readObject();
				assert o instanceof HashMap<?, ?>;
				loadedLocks = (Map<User, Set<Material>>) o;

				o = ois.readObject();
				assert o instanceof HashMap<?, ?>;
				loadedFriends = (Map<User, Set<User>>) o;

				ois.close();

			} catch (ClassNotFoundException e) {
				plugin.getOutput().printException("Corrupted protection file.", e);
				ois.close();
				plugin.getProtectionFile().delete();
				loadedProtections = (loadedProtections != null) ? loadedProtections : new HashMap<User, Set<Material>>(initialUserCapacity);
				loadedLocks = (loadedLocks != null) ? loadedLocks : new HashMap<User, Set<Material>>(initialUserCapacity);
				loadedFriends = (loadedFriends != null) ? loadedFriends : new HashMap<User, Set<User>>(initialUserCapacity);
			} finally {
				protections = loadedProtections;
				locks = loadedLocks;
				friends = loadedFriends;
			}
		} else {
			protections = new HashMap<User, Set<Material>>(initialUserCapacity);
			locks = new HashMap<User, Set<Material>>(initialUserCapacity);
			friends = new HashMap<User, Set<User>>(initialUserCapacity);
		}
	}

	private Set<Material> getAllowedMaterials() {
		List<String> allowed = Setting.PROTECTION_ALLOWED_MATERIALS.get();
		HashSet<Material> result = new HashSet<Material>(allowed.size(), 1);
		Material m;
		for (String materialName : allowed) {
			try {
				m = Material.parseMaterial(materialName);
				result.add(m);
			} catch (IllegalArgumentException e) {
			}
		}
		return result;
	}

	private Set<Material> getDisallowedMaterials() {
		List<String> disallowed = Setting.PROTECTION_DISALLOWED_MATERIALS.get();
		HashSet<Material> result = new HashSet<Material>(disallowed.size(), 1);
		Material m;
		for (String materialName : disallowed) {
			try {
				m = Material.parseMaterial(materialName);
				result.add(m);
			} catch (IllegalArgumentException e) {
			}
		}
		return result;
	}

	private Set<Material> getAutoProtectedMaterials() {
		List<String> autoProtected = Setting.PROTECTION_AUTO_PROTECT_MATERIALS.get();
		HashSet<Material> result = new HashSet<Material>(autoProtected.size(), 1);
		Material m;
		for (String materialName : autoProtected) {
			m = Material.parseMaterial(materialName);
			if (m != null) {
				result.add(m);
			}
		}
		return result;
	}

	private void save() {
		try {
			ObjectOutputStream oos = new ObjectOutputStream(new FileOutputStream(plugin.getProtectionFile(), false));
			oos.writeObject(protections);
			oos.writeObject(locks);
			oos.writeObject(friends);
			oos.close();
		} catch (FileNotFoundException e) {
			plugin.getOutput().printException("Error while saving protection file", e);
			return;
		} catch (IOException e) {
			plugin.getOutput().printException("Error while saving protection file", e);
			return;
		}
	}

	private void doAction(ProtectAction protectAction) {
		switch (protectAction.getActionType()) {
		case PROTECT:
			protect(protectAction.getOwner(), protectAction.getMaterial());
			return;
		case UNPROTECT:
			unprotect(protectAction.getOwner(), protectAction.getMaterial());
			return;
		case LOCK:
			lock(protectAction.getOwner(), protectAction.getMaterial());
			return;
		case UNLOCK:
			unlock(protectAction.getOwner(), protectAction.getMaterial());
			return;
		case FRIEND:
			friend(protectAction.getOwner(), protectAction.getUser());
			return;
		case UNFRIEND:
			unfriend(protectAction.getOwner(), protectAction.getUser());
			return;
		case DROP:
			dropUserData(protectAction.getUser());
			return;
		default:
			throw new IllegalArgumentException();
		}
	}

	private void dropUserData(User user) {
		protections.remove(user);
		locks.remove(user);
		friends.remove(user);
		Iterator<Set<User>> iterator = friends.values().iterator();
		Set<User> list;
		while (iterator.hasNext()) {
			list = iterator.next();
			list.remove(user);
		}
	}

	private void protect(User owner, Material material) {
		if (protections.get(owner) == null) {
			protections.put(owner, new HashSet<Material>(initialMaterialCapacity));
		}
		protections.get(owner).add(material);
		return;
	}

	private void unprotect(User owner, Material material) {
		if (protections.get(owner) == null) {
			protections.put(owner, new HashSet<Material>(initialMaterialCapacity));
		}
		protections.get(owner).remove(material);
		return;
	}

	private void lock(User owner, Material material) {
		if (locks.get(owner) == null) {
			locks.put(owner, new HashSet<Material>(initialMaterialCapacity));
		}
		locks.get(owner).add(material);
		return;
	}

	private void unlock(User owner, Material material) {
		if (locks.get(owner) == null) {
			locks.put(owner, new HashSet<Material>(initialMaterialCapacity));
		}
		locks.get(owner).remove(material);
		return;
	}

	private void friend(User owner, User user) {
		if (friends.get(owner) == null) {
			friends.put(owner, new HashSet<User>(initialUserCapacity));
		}
		friends.get(owner).add(user);
		return;
	}

	private void unfriend(User owner, User user) {
		if (friends.get(owner) == null) {
			friends.put(owner, new HashSet<User>(initialUserCapacity));
		}
		friends.get(owner).remove(user);
		return;
	}

	private boolean isAllowed(Material material) {
		if (material.isAny()) return true;
		if (disallowedMaterials.contains(material)) return false;
		else return (allowedMaterials.isEmpty() || allowedMaterials.contains(material));
	}

	private boolean isAutoProtected(Material material) {
		if (autoProtectedMaterials.contains(material)) return true;
		if (autoProtectedMaterials.contains(Material.ANY)) return true;
		if (!material.isBlock()) {
			if (Animals.class.isAssignableFrom(material.getEntityType().getEntityClass())) {
				return Setting.PROTECTION_AUTO_PROTECT_OWNED_ANIMALS.get();
			} else if (!LivingEntity.class.isAssignableFrom(material.getEntityType().getEntityClass())) {
				return Setting.PROTECTION_AUTO_PROTECT_OWNED_ENTITIES.get();
			} else {
				return false;
			}
		} else {
			return Setting.PROTECTION_AUTO_PROTECT_OWNED_BLOCKS.get();
		}
	}

	/**
	 * Checks whether a user can access an Ownable consisting of the specified material that is owned by the specified owner.
	 *
	 * @param owner
	 *            the owner
	 * @param material
	 *            the material
	 * @param user
	 *            the user
	 * @return true, if allowed
	 */
	public boolean hasAccess(User owner, Material material, User user) {
		if (isListLocked(owner, material)) return false;
		if (isFriend(owner, user)) return true;
		if (isAutoProtected(material)) return false;
		if (!isAllowed(material)) return true;
		if (isListProtected(owner, material)) return false;
		return true;
	}

	private boolean isListLocked(User owner, Material material) {
		Set<Material> lockList = locks.get(owner);
		if (lockList == null) return false;
		if (lockList.contains(material)) return true;
		return lockList.contains(Material.ANY);
	}

	/**
	 * Checks whether the user has locked the access to the material
	 *
	 * @param owner
	 *            the owner
	 * @param material
	 *            the material
	 * @return true, if is locked
	 */
	public boolean isLocked(User owner, Material material) {
		if (!isAllowed(material)) return false;
		return isListLocked(owner, material);
	}

	/**
	 * Checks whether owner has declared user as a friend. Please note that "user if friend of owner" doesn't imply "owner is friend of user"
	 *
	 * @param owner
	 *            the owner
	 * @param user
	 *            the user
	 * @return true, if is friend
	 */
	public boolean isFriend(User owner, User user) {
		Set<User> friendList = friends.get(owner);
		if (friendList == null) return false;
		return (friendList.contains(user));
	}

	private boolean isListProtected(User owner, Material material) {
		Set<Material> protectionList = protections.get(owner);
		if (protectionList == null) return false;
		if (protectionList.contains(material)) return true;
		return protectionList.contains(Material.ANY);
	}

	/**
	 * Checks whether the material is protected
	 * 
	 * @param owner
	 *            the owner
	 * @param material
	 *            the material
	 * @return True, if material is protected by owner
	 */
	public boolean isProtected(User owner, Material material) {
		if (isAutoProtected(material)) return true;
		if (!isAllowed(material)) return false;
		return isListProtected(owner, material);
	}

	/**
	 * Gets an immutable Set of friends for a user
	 * 
	 * @param owner
	 *            the User
	 * @return immutable Set, never null
	 */
	public Set<User> getFriends(User owner) {
		Set<User> result = friends.get(owner);
		if (result == null) result = new HashSet<User>();
		return Collections.unmodifiableSet(result);
	}

	/**
	 * Gets an immutable Set of locked materials for a user
	 * 
	 * @param owner
	 *            the User
	 * @return immutable Set, never null
	 */
	public Set<Material> getLocks(User owner) {
		Set<Material> result = locks.get(owner);
		if (result == null) result = new HashSet<Material>();
		return Collections.unmodifiableSet(result);
	}

	/**
	 * Gets an immutable Set of protected materials for a user
	 * 
	 * @param owner
	 *            the User
	 * @return immutable Set, never null
	 */
	public Set<Material> getProtections(User owner) {
		Set<Material> result = protections.get(owner);
		if (result == null) result = new HashSet<Material>();
		return Collections.unmodifiableSet(result);
	}

	/**
	 * Enqueue a ProtectAction.
	 *
	 * @param protectAction
	 *            the ProtectAction
	 */
	public void enqueue(ProtectAction protectAction) {
		synchronized (LOCK) {
			queue.add(protectAction);
			LOCK.notifyAll();
		}
	}

	/**
	 * Needs to be called to make protection thread empty the queue before program ends
	 */
	public void disable() {
		synchronized (LOCK) {
			disable = true;
			LOCK.notifyAll();
		}
	}

	@Override
	public void run() {
		ProtectAction protectAction;
		while (!disable || !queue.isEmpty()) {
			if (!queue.isEmpty()) {
				protectAction = queue.remove();
				doAction(protectAction);
			} else {
				try {
					synchronized (LOCK) {
						while (!disable && queue.isEmpty()) {
							LOCK.wait();
						}
					}
				} catch (InterruptedException e) {
					plugin.getOutput().printException("Database thread has been interrupted.", e);
				}
			}
		}

		save();
	}
}
