package de.pheasn.blockown.database;

import java.text.SimpleDateFormat;
import java.util.Date;

import de.pheasn.blockown.Ownable;
import de.pheasn.blockown.User;

public class DatabaseAction {

	private final User user;
	private final Ownable ownable;
	private final DatabaseActionType databaseActionType;
	private final long timeStamp;

	/**
	 * Instantiates a new DatabaseAction.
	 *
	 * @param databaseActionType
	 *            the DatabaseActionType
	 * @param ownable
	 *            the ownable
	 * @param user
	 *            the user, may be null or left out if databaseActionType is UNOWN
	 */
	public DatabaseAction(DatabaseActionType databaseActionType, Ownable ownable, User user) {
		this.user = user;
		this.ownable = ownable;
		this.databaseActionType = databaseActionType;
		this.timeStamp = System.currentTimeMillis();
	}

	/**
	 * May only be used with DatabaseActionType UNOWN
	 * 
	 * @param databaseActionType
	 *            the DatabaseActionType
	 * @param ownable
	 *            the ownable
	 */
	public DatabaseAction(DatabaseActionType databaseActionType, Ownable ownable) {
		this(databaseActionType, ownable, null);
	}

	/**
	 * May only be used with DatabaseActionType DROP
	 * 
	 * @param databaseActionType
	 *            the DatabaseActionType
	 * @param user
	 *            the user whos data will be dropped
	 */
	public DatabaseAction(DatabaseActionType databaseActionType, User user) {
		this(databaseActionType, null, user);
	}

	public User getUser() {
		return user;
	}

	public Ownable getOwnable() {
		return ownable;
	}

	public long getTimeStamp() {
		return timeStamp;
	}

	public DatabaseActionType getActionType() {
		return databaseActionType;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((databaseActionType == null) ? 0 : databaseActionType.hashCode());
		result = prime * result + ((ownable == null) ? 0 : ownable.hashCode());
		result = prime * result + (int) (timeStamp ^ (timeStamp >>> 32));
		result = prime * result + ((user == null) ? 0 : user.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj) return true;
		if (obj == null) return false;
		if (getClass() != obj.getClass()) return false;
		DatabaseAction other = (DatabaseAction) obj;
		if (databaseActionType != other.databaseActionType) return false;
		if (ownable == null) {
			if (other.ownable != null) return false;
		} else if (!ownable.equals(other.ownable)) return false;
		if (timeStamp != other.timeStamp) return false;
		if (user == null) {
			if (other.user != null) return false;
		} else if (!user.equals(other.user)) return false;
		return true;
	}

	@Override
	public String toString() {
		return "DatabaseAction [user=" + user
				+ ", ownable=" + ownable
				+ ", databaseActionType=" + databaseActionType
				+ ", timeStamp=" + new SimpleDateFormat().format(new Date(timeStamp))
				+ "]";
	}
}
