package de.pheasn.blockown.database;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;

import org.bukkit.Bukkit;

import de.pheasn.blockown.Ownable;
import de.pheasn.blockown.User;

public class Cache {

	private final File dumpFile;
	private final CachedDatabase database;

	private final Map<Ownable, User> cache;
	private final Map<Ownable, User> unsubmitted;

	Cache(CachedDatabase database) {
		this.database = database;
		String dataFolderPath;
		try {
			dataFolderPath = database.getPlugin().getDataFolder().getPath();
		} catch (NullPointerException e) {
			// Neccessary for tests
			dataFolderPath = ".";
		}
		dumpFile = new File(dataFolderPath, "dumpFile.dat");
		if (dumpFile.exists()) {
			unsubmitted = restoreDumpedCache();
			cache = initializeCache();
			cache.putAll(unsubmitted);
		} else {
			unsubmitted = initializeCache();
			cache = initializeCache();
		}
	}

	/**
	 * Creates a new HashMap with an appropriate capacity
	 * 
	 * @return the HashMap
	 */
	private Map<Ownable, User> initializeCache() {
		int initialCapacity;
		try {
			initialCapacity = Bukkit.getOfflinePlayers().length * 1000;
		} catch (NullPointerException e) {
			// Neccessary for testing
			initialCapacity = 0;
		}
		if (initialCapacity == 0) initialCapacity = 5000;
		return new HashMap<Ownable, User>(initialCapacity);
	}

	/**
	 * Restores cache from dump file and deletes file afterwards. If unsuccessful, returns empty cache
	 * 
	 * @return the restored cache
	 */
	@SuppressWarnings("unchecked")
	private Map<Ownable, User> restoreDumpedCache() {
		Map<Ownable, User> result = null;
		try {
			ObjectInputStream ois = new ObjectInputStream(new FileInputStream(dumpFile));
			Object o = ois.readObject();
			ois.close();
			if (o instanceof Map<?, ?>) {
				assert ((Map<?, ?>) o).keySet().iterator().next() instanceof Ownable;
				assert ((Map<?, ?>) o).values().iterator().next() instanceof User;
				result = (Map<Ownable, User>) o;
			}
			dumpFile.delete();
		} catch (FileNotFoundException e) {
			database.getPlugin().getOutput().printException("Error restoring dumped cache.", e);
		} catch (IOException e) {
			database.getPlugin().getOutput().printException("Error restoring dumped cache.", e);
		} catch (ClassNotFoundException e) {
			database.getPlugin().getOutput().printException("Error restoring dumped cache.", e);
		}
		return (result == null) ? initializeCache() : result;
	}

	User getOwner(Ownable ownable) {
		User owner = cache.get(ownable);
		if (owner == null) {
			owner = database.getDatabaseOwner(ownable);
			cache.put(ownable, owner);
		}
		return owner;
	}

	boolean doAction(DatabaseAction databaseAction) {
		if (databaseAction.getActionType() == DatabaseActionType.UNOWN || databaseAction.getUser() == null || databaseAction.getUser().isNobody()) {
			cache.put(databaseAction.getOwnable(), User.nobody);
			unsubmitted.put(databaseAction.getOwnable(), User.nobody);
			return true;
		} else if (databaseAction.getActionType() == DatabaseActionType.OWN) {
			cache.put(databaseAction.getOwnable(), databaseAction.getUser());
			unsubmitted.put(databaseAction.getOwnable(), databaseAction.getUser());
			return true;
		} else if (databaseAction.getActionType() == DatabaseActionType.DROP) {
			dropUserData(databaseAction.getUser());
			return true;
		} else {
			database.getPlugin().getOutput().printException(new IllegalArgumentException());
			return false;
		}
	}

	/**
	 * Flush all cache data to the database
	 */
	void flush() {
		if (!database.flushDatabase(unsubmitted)) {
			try {
				dumpFile.createNewFile();
				ObjectOutputStream oos = new ObjectOutputStream(new FileOutputStream(dumpFile, false));
				oos.writeObject(unsubmitted);
				oos.close();
			} catch (IOException e) {
				database.getPlugin().getOutput().printException(e);
			}
			database.getPlugin().reportDumpedCache();
		} else {
			unsubmitted.clear();
		}
	}

	void dropUserData(User user) {
		if (user == null) return;
		Iterator<Entry<Ownable, User>> iterator = cache.entrySet().iterator();
		Entry<Ownable, User> entry;
		while (iterator.hasNext()) {
			entry = iterator.next();
			if (user.equals(entry.getValue())) {
				iterator.remove();
			}
		}
		iterator = unsubmitted.entrySet().iterator();
		while (iterator.hasNext()) {
			entry = iterator.next();
			if (user.equals(entry.getValue())) {
				iterator.remove();
			}
		}
		database.dropDatabaseUserData(user);
	}

}
