package de.pfabulist.loracle.license;

import de.pfabulist.frex.Frex;
import de.pfabulist.loracle.maven.Coordinates;
import de.pfabulist.loracle.text.Normalizer;
import de.pfabulist.roast.collection.Map_;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;

import javax.annotation.Nullable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.TreeMap;
import java.util.regex.Pattern;
import java.util.stream.Stream;

import static de.pfabulist.frex.Frex.txt;
import static de.pfabulist.roast.NonnullCheck.n_;
import static de.pfabulist.roast.NonnullCheck.n_or;

/**
 * Copyright (c) 2006 - 2016, Stephan Pfab
 * SPDX-License-Identifier: BSD-2-Clause
 */

@SuppressFBWarnings( { "URF_UNREAD_PUBLIC_OR_PROTECTED_FIELD", "UUF_UNUSED_PUBLIC_OR_PROTECTED_FIELD" } )
public class LOracleData {

    private static Pattern hasOr = Frex.or( txt( " "), txt( "(") ).then( txt( "or ") ).buildCaseInsensitivePattern();

    public Optional<More> getSingle( SingleLicense loLicenseId ) {
        return Optional.ofNullable( singles.get( loLicenseId.toString() ));
    }

    public Optional<LicenseException> getExistingExclude( LicenseException exclude ) {
        if ( Map_.r_( licenseExceptions).get_o( exclude.getName() ).isPresent()) {
            return Optional.of( exclude );
        }
        return Optional.empty();
    }

    public Optional<More> getExistingLicense( LicenseID lid ) {
        if( lid instanceof SingleLicense ) {
            return getSingle( (SingleLicense) lid );
        }

        if( lid instanceof ModifiedSingleLicense ) {
            ModifiedSingleLicense modi = (ModifiedSingleLicense) lid;
            SingleLicense base = modi.getBaseLicense();
            Optional<More> existinBase = getSingle( base );
            if( !existinBase.isPresent() ) {
                throw new IllegalStateException( "modified license on non existing single license " + lid );
            }

            @Nullable More more = composites.get( lid.getId() );
            if( more != null ) {
                return Optional.of( more );
            }

            more = new More( existinBase.get().getDefinitionSource() );
            composites.put( lid.getId(), more );

            return Optional.of( more );
        }

        throw new IllegalStateException( "todo" );
    }

    @SuppressFBWarnings( "URF_UNREAD_PUBLIC_OR_PROTECTED_FIELD" )
    public static class More {
        public LicenseAttributes attributes;
        public List<String> urls = new ArrayList<>();
        public List<String> longNames = new ArrayList<>();
        public List<Coordinates> specific = new ArrayList<>();
        public Set<String> couldbeName = new HashSet<>();
        public Set<String> couldbeUrl = new HashSet<>();

        public More( LicenseDefinitionSource src ) {
            attributes = new LicenseAttributes();
            attributes.setSource( src );
        }

        public void addLongName( LoLongName name ) {
            if ( !longNames.contains( name.toString() )) {
                longNames.add( name.toString() );
            }
        }

        public LicenseDefinitionSource getDefinitionSource() {
            return attributes.getDefinitionSource();
        }
    }

    private Map<String, More> singles = new TreeMap<>( String::compareTo );
    //new HashMap<>();
    private Map<String, Boolean> licenseExceptions = new HashMap<>();
    private Map<String, More> composites = new HashMap<>();
    private List<String> tooSimpleLongNames = new ArrayList<>();


    public boolean hasSingleLicense( SingleLicense lic ) {
        return singles.containsKey( lic.toString() );
        //return Optional.ofNullable( singles.get( lic.toString() );
    }

    public Optional<String> whyIsBadLongName( String reduced ) {
        try {
            throwIfBadLongName( reduced );
        } catch( IllegalArgumentException e ) {
            return Optional.of( e.getMessage());
        }

        return Optional.empty();

    }

    public void throwIfBadLongName( String reduced ) {
            String longName = Normalizer.reduce( reduced );

            if( hasOr.matcher( longName ).find() ) {
                throw new IllegalArgumentException( "has on 'or' in it" );
            }

            singles.forEach( ( license, more ) -> {
                more.longNames.forEach( ln -> {
                    if( ln.equals( longName ) ) {
                        throw new IllegalArgumentException( "old long name: " + ln + "   (" + license + ")");
                    }
                } );
            } );

    }

    public void throwIfBadLongName( SingleLicense id, String reduced ) {
        String longName = Normalizer.reduce( reduced );

        if( hasOr.matcher( longName ).find() ) {
            throw new IllegalArgumentException( "has on 'or' in it" );
        }

        singles.forEach( ( license, more ) -> {
            if ( !license.endsWith( id.getId() )) {
                more.longNames.forEach( ln -> {
                    if( ln.equals( longName ) ) {
                        throw new IllegalArgumentException( "old long name " + ln );
                    }
                } );
            }
        } );

    }


    public boolean isUsed( SingleLicense id, LoLongName name ) {
        try {
            singles.forEach( ( license, more ) -> {
                if( !id.toString().equals( license ) ) {
                    more.longNames.forEach( ln -> {
                        if( ln.equals( name.toString() ) ) {
                            throw new IllegalArgumentException( "old long name " + ln );
                        }
                    } );
                }
            } );
        } catch( IllegalArgumentException e ) {
            return false;
        }

        return true;

    }

    public void addDejaSingle( SingleLicense li ) {
        if ( singles.get( li.toString() ) != null ) {
            throw new IllegalArgumentException( "old single license " + li  );
        }

        //LoLongName name = LoLongName.newStd( li.toString() );
        throwIfBadLongName( li.toString() );

        More more = new More( new LicenseDefinitionSource( "d" ) );         // todo
        more.longNames.add( LoLongName.newStd( li.toString() ).toString() );
        //more.attributes.setSPDX( true );
        singles.put( li.toString(), more );
        //return more;
    }

    public void addSpdxSingle( SingleLicense li ) {
        if ( singles.get( li.toString() ) != null ) {
            //More old = singles.get( li.toString() );
            throw new IllegalArgumentException( "old single license " + li  );
        }

        //LoLongName name = LoLongName.newStd( li.toString() );
        throwIfBadLongName( li.toString() );

        More more = new More( new LicenseDefinitionSource( "s" ) );
        more.longNames.add( LoLongName.newStd( li.toString() ).toString() );
        //more.attributes.setSPDX( true );
        singles.put( li.toString(), more );
        //return more;
    }

    public void addLongName( SingleLicense id, String str ) {
        String reduced = Normalizer.reduce( str );
        throwIfBadLongName( id, reduced );

        @Nullable More more = singles.get( id.toString() );

        if ( more == null ) {
            throw new IllegalArgumentException( "not a single license " + id );
        }

        more.longNames.add( reduced );

    }

    public void addUrl( SingleLicense id, String url ) {
        @Nullable More more = singles.get( id.toString() );

        if ( more == null ) {
            throw new IllegalArgumentException( "not a single license " + id );
        }

        more.urls.add( url );
    }

    public void addException( LicenseException ex ) {
        licenseExceptions.put( ex.toString(), true );
    }

    public Stream<Map.Entry<String,More>> getSingleStream() {
        return singles.entrySet().stream();
    }

    public LicenseAttributes getAttributes( SingleLicense id ) {
        @Nullable More more = singles.get( id.toString() );

        if ( more == null ) {
            throw new IllegalArgumentException( "not a single license " + id );
        }

        return more.attributes;
    }


}
