package de.pfabulist.loracle.license;

import de.pfabulist.kleinod.frex.Frex;

import java.util.Locale;
import java.util.regex.Pattern;

import static de.pfabulist.kleinod.frex.Frex.txt;

/**
 * Copyright (c) 2006 - 2017, Stephan Pfab
 * SPDX-License-Identifier: BSD-2-Clause
 */

// allow strings, lowercase, numbers, -, .
// and not 'and', 'or', 'with'
public class SingleLicense implements LicenseID {

    private final LicenseName name;

    static public boolean isValidString( String str ) {
        return good.matcher( str ).matches() && !blocked.matcher( str ).find();
    }

    public SingleLicense( String id ) {
        name = new LicenseName( id );
        if ( !isValid( id )) {
            throw new IllegalArgumentException( "str passes LicenseName validation but not SingleLicense validation " + id );
        }
    }

    public boolean isValid( String str ) {
        return isValidString( str );
    }

    static Frex goodFrex =
            Frex.or( Frex.alphaLower(), Frex.number() ).
                    then( Frex.or( Frex.alphaLower(),
                                   Frex.number(),
                                   txt( '-' ),
                                   txt( '.' ) ).
                            zeroOrMore() );

    static public Frex getGoodFrex() {
        return goodFrex;
    }

    static Pattern good = goodFrex.buildPattern();

    static Frex badFrex = Frex.or( Frex.fullWord( "or" ), Frex.fullWord( "and" ), Frex.fullWord( "with" ) );

    public Frex getBadFrex() { return badFrex; }

    static Pattern blocked = badFrex.buildCaseInsensitivePattern();

    public static SingleLicense newStd( String in ) {
        String lower = in.toLowerCase( Locale.US );
        return new SingleLicense( lower );
    }

    @Override
    public LicenseName getId() {
        return name;
    }

    @Override
    public int compareTo( LicenseID o ) {
        return toString().compareTo( o.toString() ) ;
    }

    @Override
    public boolean equals( Object o ) {
        if( this == o ) return true;
        if( o == null || getClass() != o.getClass() ) return false;

        SingleLicense that = (SingleLicense) o;

        return name.equals( that.name );
    }

    @Override
    public int hashCode() {
        return name.hashCode();
    }

    @Override
    public String toString() {
        return name.toString();
    }

    @Override
    public boolean isSingle() {
        return true;
    }

    @Override
    public boolean isAlternative() {
        return false;
    }
}
