package de.pfabulist.loracle.license;

import de.pfabulist.kleinod.frex.Frex;
import de.pfabulist.kleinod.frex.FrexMatcher;
import de.pfabulist.kleinod.frex.FrexPattern;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;

import javax.annotation.Nullable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static de.pfabulist.kleinod.frex.Frex.txt;
import static de.pfabulist.roast.types.NonnullCheck.n_;

/**
 * Copyright (c) 2006 - 2017, Stephan Pfab
 * SPDX-License-Identifier: BSD-2-Clause
 *
 * a single license can be modified by adding (one or several) exception(s)
 * or a plus for "or later"
 */
public class ModifiedSingleLicense implements LicenseID {

    private final LicenseID base;
    private final boolean orLater;
    private final List<LicenseException> exception;

    public ModifiedSingleLicense( LicenseID singleLicense, boolean orLater, Optional<LicenseException> exception ) {
        this.exception = exception.map( Arrays::asList ).orElseGet( Collections::emptyList );
        this.base = singleLicense;
        this.orLater = orLater;
    }

    enum NamePatternVars {
        name,
        plus,
        exception
    }


    static private FrexPattern namePattern =
            SingleLicense.getGoodFrex().var( NamePatternVars.name ).
                    then( Frex.whitespace().zeroOrMore()).
                    then( txt( '+' ).var( NamePatternVars.plus ).zeroOrOnce() ).
                    then( Frex.whitespace().oneOrMore().
                            then( txt( "WITH" ) ).
                            then( Frex.whitespace().oneOrMore() ).
                            then( SingleLicense.getGoodFrex().var( NamePatternVars.exception ) ).zeroOrOnce() ).
                    buildCaseInsensitiveFrexPattern();

    static private FrexPattern altNamePattern =
            Frex.any().oneOrMore().lazy().var( NamePatternVars.name ).
                    then( Frex.whitespace().zeroOrMore()).
                    then( txt( '+' ).var( NamePatternVars.plus ).zeroOrOnce() ).
                    then( Frex.whitespace().oneOrMore().
                            then( txt( "WITH" ) ).
                            then( Frex.whitespace().oneOrMore() ).
                            then( SingleLicense.getGoodFrex().var( NamePatternVars.exception ) ).zeroOrOnce() ).
                    buildCaseInsensitiveFrexPattern();

    public static LicenseID getAlternativeOrModifiedLicense( String nameExpr ) {

        FrexMatcher matcher = altNamePattern.getMatcher_ot( nameExpr );

//        Matcher matcher = namePattern.matcher( nameExpr );
//        if( !matcher.matches() ) {
//            // todo other licenses
//            throw new IllegalArgumentException( "no such license: " + nameExpr );
//        }

        AlternativeLicense sli = AlternativeLicense.newStd( matcher.get_ot( NamePatternVars.name ));

//        String name = _nn( matcher.group( "name" ) );
//        SingleLicense license = LOracle2.getExistingSingle( lOracle, name ).
//                orElseThrow( () -> new IllegalArgumentException( "no such license: " + name ) );

        

        boolean plus = matcher.get( NamePatternVars.plus ).isPresent(); //matcher.group( "plus" ) != null;
        Optional<LicenseException> ex = matcher.get( NamePatternVars.exception ).map( LicenseException::newStd );

        if ( plus || ex.isPresent() ) {
            return new ModifiedSingleLicense( sli, plus, ex );
        }

        return sli;
//        Optional<LicenseException> exception = Optional.ofNullable( matcher.group( "exception" ) ).
//                map( str -> LOracle2.getExceptionOrThrow( lOracle, str ) );
//
//        if ( plus || exception.isPresent() ) {
//            throw new IllegalArgumentException( "damn" );
//        }
//        return license;
//        //return lOracle.getOrLater( license, plus, exception );

    }

    public static LicenseID getSingleOrModifiedLicense( String nameExpr ) {

        FrexMatcher matcher = namePattern.getMatcher_ot( nameExpr );

//        Matcher matcher = namePattern.matcher( nameExpr );
//        if( !matcher.matches() ) {
//            // todo other licenses
//            throw new IllegalArgumentException( "no such license: " + nameExpr );
//        }

        SingleLicense sli = SingleLicense.newStd( matcher.get_ot( NamePatternVars.name ));

//        String name = _nn( matcher.group( "name" ) );
//        SingleLicense license = LOracle2.getExistingSingle( lOracle, name ).
//                orElseThrow( () -> new IllegalArgumentException( "no such license: " + name ) );



        boolean plus = matcher.get( NamePatternVars.plus ).isPresent(); //matcher.group( "plus" ) != null;
        Optional<LicenseException> ex = matcher.get( NamePatternVars.exception ).map( LicenseException::newStd );

        if ( plus || ex.isPresent() ) {
            return new ModifiedSingleLicense( sli, plus, ex );
        }

        return sli;
//        Optional<LicenseException> exception = Optional.ofNullable( matcher.group( "exception" ) ).
//                map( str -> LOracle2.getExceptionOrThrow( lOracle, str ) );
//
//        if ( plus || exception.isPresent() ) {
//            throw new IllegalArgumentException( "damn" );
//        }
//        return license;
//        //return lOracle.getOrLater( license, plus, exception );

    }

    @Override
    public LicenseName getId() {

        return new LicenseName( base + (orLater ? "+" : "" ) + //exception.map( e -> " with " + e ).orElse( "" );
                (exception.size() > 0 ? ( " with " + exception.get( 0 ).getName() ) : ""));
    }

    @Override
    @SuppressFBWarnings( "NP_METHOD_PARAMETER_TIGHTENS_ANNOTATION" )
    public boolean equals( @Nullable Object o ) {
        if( this == o ) { return true; }
        if( o == null || getClass() != o.getClass() ) { return false; }

        ModifiedSingleLicense that = (ModifiedSingleLicense) o;

        if( orLater != that.orLater ) { return false; }
        if( !base.equals( that.base ) ) { return false; }
        return exception.equals( that.exception );

    }

    @Override
    public int hashCode() {
        int result = base.hashCode();
        result = 31 * result + ( orLater ? 1 : 0 );
        result = 31 * result + exception.hashCode();
        return result;
    }

    @Override
    public String toString() {
        return getId().toString();
    }

    // todo to list
    public Optional<LicenseException> getException() {
        return exception.size() > 0 ? Optional.of( exception.get(0)) : Optional.empty();
    }

    public String getBase() {
        return base.getId().toString();
    }

    public LicenseID getBaseLicense() {
        return base;
    }

    @Override
    public int compareTo( LicenseID o ) {
        return toString().compareTo( o.toString() ) ;
    }

    @Override
    public boolean isAlternative() {
        return base.isAlternative();
    }
}
