package de.pfabulist.lindwurm.toti;

import com.esotericsoftware.minlog.Log;
import de.pfabulist.lindwurm.stellvertreter.PasswordHash;
import de.pfabulist.lindwurm.stellvertreter.Stellvertreter;
import de.pfabulist.unchecked.Filess;
import org.json.JSONObject;

import java.io.Console;
import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;

import static de.pfabulist.kleinod.text.Strings.getBytes;
import static de.pfabulist.kleinod.text.Strings.newString;
import static de.pfabulist.unchecked.Unchecked.u;

/**
 * ** BEGIN LICENSE BLOCK *****
 * BSD License (2 clause)
 * Copyright (c) 2006 - 2015, Stephan Pfab
 * All rights reserved.
 * <p>
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * <p>
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL Stephan Pfab BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * **** END LICENSE BLOCK ****
 */

public class CollectConf {

    private final Conf conf;
    private final boolean existing;
    private final Console console = System.console();
    private final Path host;

    private CollectConf( Conf conf, boolean existing, Path host ) {
        this.conf = conf;
        this.existing = existing;
        this.host = host;
    }

    public static CollectConf existingOrDefault( Path host ) {

        if( !Stellvertreter.isHost( host ) ) {
            return new CollectConf( new Conf(), false, host );
        }

        Path confFile = getIcebergConfPath( host );

        if( !Files.exists( confFile ) ) {
            return new CollectConf( new Conf(), true, host );
        }

        try {
            String content = newString( Files.readAllBytes( confFile ) );
            JSONObject jsonObject = new JSONObject( content );

            return new CollectConf( Conf.valueOf( jsonObject ), true, host );
        } catch( IOException e ) {
            Log.warn( "toti conf corrupt", e );
            return new CollectConf( new Conf(), true, host );
        }
    }

    private static Path getIcebergConfPath( Path host ) {
        return host.resolve( ".toti.conf" );
    }

    public CollectConf getUserInput() {
        if( existing ) {
            System.out.println( "opening an existing iceberg" );
        } else if( !isEmpty( host ) ) {
            System.out.println( "[error] not an existing host but in use for other purpose (not empty)" );
            System.exit( 1 );
        } else {
            System.out.println( "creating a new iceberg ..." );
        }
        conf.passwd = getPasswd();
        conf.volume = getVolume();
        conf.logToConsole = getLogToConsole();
        conf.logToFile = getBoolean( "log to file", conf.logToFile ); // dropbox

        if( !existing ) {
            conf.elsewhere = getElseWhere();
        }

        conf.spaceLimit = getSpaceLimit();

        return this;
    }

    private long getSpaceLimit() {
        if( existing ) {
            if( !hasElsewheres( host ) ) {
                return -1;
            }
        } else {
            if( conf.elsewhere == null ) {
                return -1;
            }
        }

        while ( true ) {
            String space = console.readLine( "max local space [" + conf.spaceLimit + "]> " );

            if( space.length() == 0 ) {
                return conf.spaceLimit;
            }

            try {
                long sp = Long.valueOf( space );
                return sp;
            } catch( NumberFormatException e ) {
                System.out.println("a long please");
            }
        }
    }

    private Path getElseWhere() {
        while( true ) {
            String el = console.readLine( "Elsewhere [none]> " );

            if( el.length() == 0 ) {
                return null;
            }

            Path els = Paths.get( el );

            if( Files.exists( els ) && Files.isDirectory( els ) && isEmpty( els ) ) {
                return els;
            }

            System.out.println( "else must be an existing empty dir" );
        }
    }

    private boolean getLogToConsole() {
        return getBoolean( "log to console", conf.logToConsole );
    }

    private boolean getBoolean( String message, boolean dflt ) {
        while( true ) {
            String l = console.readLine( message + "[" + ( dflt ? "Yn]" : "yN]> " ) );

            if( l.length() > 0 ) {
                if( l.equals( "y" ) ) {
                    return true;
                }

                if( l.equals( "n" ) ) {
                    return false;
                }

                System.out.println( "'y' or 'n' or enter" );
            } else {
                return dflt;
            }
        }
    }

    private String getVolume() {

        String vol = conf.volume;
        if( conf.volume == null ) {
            vol = host.getFileName().toString();
        }

        while( true ) {

            String v = console.readLine( "volume [" + vol + "]> " );
            if( v.length() > 0 ) {
                vol = v;
            }

            if( Files.exists( Paths.get( "/Volumes/" + vol ) ) ) {
                System.out.println( "[error] volume in use choose an other" );
            } else {
                break;
            }
        }

        return vol;
    }

    private String getPasswd() {
        if( existing ) {
            return getExistingPassword();
        }

        return getNewPassword();
    }

    private String getNewPassword() {
        String passwd = "";
        while( true ) {
            passwd = console.readLine( "choose password> " );

            if( passwd.length() < 10 ) {
                System.out.println( "password too short (<10)" );
            } else {
                String passwd2 = console.readLine( "repeat         > " );
                if( passwd.equals( passwd2 ) ) {
                    return passwd;
                }
                System.out.println( "[ERROR] passwords not equal. Try again" );
            }
        }
    }

    private String getExistingPassword() {
        String passwd;
        while( true ) {
            passwd = console.readLine( "Passwd: " );

            if( !Files.exists( host.resolve( ".svsanity" ) ) ) {
                System.out.printf( "[WARN] not possible to validate password. Lets hope its correct." );
                return passwd;
            }

            if( new PasswordHash().same( Filess.readAllBytes( host.resolve( ".svsanity" ) ), passwd.toCharArray() ) ) {
                return passwd;
            }

            System.out.println( "[ERROR] wrong password. Try again" );
        }
    }

    public Conf build() {
        Filess.write( getIcebergConfPath( host ), getBytes( conf.toString() ) );
        return conf;
    }

    private static boolean isEmpty( Path dir ) {
        try( DirectoryStream<Path> stream = Files.newDirectoryStream( dir ) ) {
            for( Path kid : stream ) {
                return false;
            }
        } catch( IOException e ) {
            throw u( e );
        }

        return true;
    }

    private static boolean hasElsewheres( Path dir ) {
        try( DirectoryStream<Path> stream = Files.newDirectoryStream( dir ) ) {
            for( Path kid : stream ) {
                if( kid.getFileName().toString().startsWith( ".elsewhere" ) ) {
                    return true;
                }
            }
        } catch( IOException e ) {
            throw u( e );
        }

        return false;
    }

}
