package de.pfabulist.lindwurm.memoryntfs;

import de.pfabulist.kleinod.paths.Filess;
import de.pfabulist.lindwurm.eighty.*;
import de.pfabulist.lindwurm.eighty.attributes.AttributesBuilder;
import de.pfabulist.lindwurm.eighty.close.CloseableSeekableByteChannel;
import de.pfabulist.lindwurm.eighty.path.EightyPath;
import de.pfabulist.lindwurm.eighty.path.EightyPathConstraints;
import de.pfabulist.lindwurm.eighty.path.WindowsPathConstraints;
import de.pfabulist.lindwurm.eighty.watch.EightyWatcher;
import de.pfabulist.lindwurm.memory.MemoryFSBuilder;
import de.pfabulist.kleinod.collection.P;

import java.io.IOException;
import java.nio.file.*;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileAttributeView;
import java.util.*;
import java.util.stream.Stream;

import static de.pfabulist.kleinod.errors.Unchecked.runtime;
import static de.pfabulist.lindwurm.eighty.EightyUtils.get80;

/**
 * ** BEGIN LICENSE BLOCK *****
 * BSD License (2 clause)
 * Copyright (c) 2006 - 2014, Stephan Pfab
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL Stephan Pfab BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * **** END LICENSE BLOCK ****
 */
public class MemoryNTFS implements EightyFS {

    private final String name;
    private List<String> roots = new ArrayList<>();
    private Map<String,FileSystem> drives = new HashMap<>();
    private EightyPathConstraints wpc = new WindowsPathConstraints();

    public static EightyFS create( Object name, AttributesBuilder ab, Map<String, ?> env ) {
        List<Character> drives = Collections.<Character>emptyList();
        if ( env.containsKey( "drives" )) {
            drives = (List<Character>) env.get("drives");
        }

                //key -> (Object)Collections.<String>emptyList() );
        return new MemoryNTFS( (String)name, drives );
    }

    public MemoryNTFS(String name, List<Character> drives ) {
        roots.add( "C:");
        this.drives.put( "C:", MemoryFSBuilder.memoryFS().name("C").build());

        for ( Character chr : drives ) {
            char ch = chr.toString().toUpperCase().toCharArray()[0];
            if ( ch < 'A' ||  ch > 'Z') {
                throw new InvalidPathException( chr.toString(), "not a drive character");
            }

            if ( ch != 'C') {
                this.roots.add( "" + ch + ":" );
                this.drives.put( "" + ch + ":" , MemoryFSBuilder.memoryFS().name( "" + ch ).build());
            }
        }

        this.name = name;
    }


    @Override
    public boolean nameEquals(String a, String b) {
        return wpc.nameEquals( a,b );
    }

    @Override
    public int nameHash(String a) {
        return wpc.nameHash(a);
    }

    @Override
    public CloseableSeekableByteChannel newByteChannel(EightyPath path, Set<? extends OpenOption> options, FileAttribute<?>... attrs)  {
        EightyPath host = getHostPath( path );

        return get80(host.getFileSystem()).newByteChannel( host, options, attrs);
    }

    @Override
    public String getSeparators() {
        return wpc.getSeparators();
    }

    @Override
    public void createDirectory(EightyPath dir, FileAttribute<?>... attrs)  {
        EightyPath host = getHostPath(dir);

        get80(host.getFileSystem()).createDirectory( host, attrs );
    }

    @Override
    public void checkAccess(EightyPath path, AccessMode... modes)  {
        try {
            EightyPath host = getHostPath( path );
            get80(host.getFileSystem()).checkAccess( host, modes );
        } catch (RuntimeException e) {
            throw runtime(new NoSuchFileException( "files does not exist " + path ));
        }
    }

    @Override
    public Stream<Path> newDirectoryStream( EightyPath dir ) {

        EightyPath host = getHostPath( dir);

        Stream<Path> hostStream = get80(host.getFileSystem()).newDirectoryStream(host);

        return hostStream.map( hostKid -> getGuestPath( dir, hostKid));
    }



    @Override
    public void move(EightyPath source, EightyPath target, CopyOption... options)  {
        Filess.move(getHostPath(source), getHostPath(target));
    }

    @Override
    public void delete(EightyPath path)  {
        Filess.delete( getHostPath(path));
    }

    @Override
    public FileStore getFileStore(EightyPath path) {
        return Filess.getFileStore( getHostPath(path));
    }

    @Override
    public Iterable<FileStore> getFileStores() {

        List<FileStore> stores = new ArrayList<>();

        for ( String key : drives.keySet() ) {
            stores.add( drives.get(key).getFileStores().iterator().next());
        }
        return stores;
    }

    @Override
    public <V extends FileAttributeView> V getFileAttributeView( EightyPath path, Class<V> type ) {
        EightyPath host = getHostPath( path);

        return get80(host.getFileSystem()).getFileAttributeView( host, type );
    }

    @Override
    public boolean isReadOnly() {
        return false;
    }

    @Override
    public boolean isHidden(EightyPath path) {
        Path host = getHostPath(path);
        return get80(host.getFileSystem()).isHidden((EightyPath) host);
    }

    @Override
    public boolean isClosable() {
        return false;
    }

    @Override
    public boolean watchable() {
        return false;
    }

    @Override
    public boolean isReopenable() {
        return false;
    }

    @Override
    public void reopen() {

    }

    @Override
    public void setWatcher(EightyWatcher eightyFileSystem) {

    }


    @Override
    public P<Optional<String>, String> parseRootComponent(String str) {
        return wpc.parseRootComponent(str);
    }

    @Override
    public Optional<Character> findIllegalChars( String fileName ) {
        for ( int i = 0; i < fileName.length(); i++ ) {

            char ch = fileName.charAt(i);

            switch (ch) {
                case ':':
                case '?':
                case '<':
                case '>':
                case '*':
                case '|':
                case '"':
                    return Optional.of(ch);
            }

//            if ( 0 <= ch && ch < \u001f ) {
//
//            }

        }

        return Optional.empty();
    }

    @Override
    public Optional<String> allowAccess( Path path) {
        return wpc.allowAccess(path);
    }

    @Override
    public List<String> getRoots() {
        return roots;
    }

    @Override
    public void createHardLink(EightyPath normLink, EightyPath normExisting) {
        EightyPath hostLink = getHostPath( normLink );

        get80(hostLink.getFileSystem()).createHardLink(hostLink, getHostPath(normExisting));
    }

    @Override
    public void createSymLink( EightyPath link, EightyPath existing, FileAttribute<?>[] attrs) {
        EightyPath hostLink = getHostPath( link );

        if ( !existing.isAbsolute() ) {
            get80(hostLink.getFileSystem()).createSymLink( hostLink, getRelHost( hostLink.getFileSystem(), existing ), attrs );
            return;
        }

        EightyPath target = getHostPath( existing);
        get80(hostLink.getFileSystem()).createSymLink( hostLink, target, attrs );

//        return;


//        get80( hostLink.getFileSystem()).createSymLink(hostLink, existing, attrs);
    }


    private EightyPath getRelHost( FileSystem hostFS, EightyPath guest ) {
        EightyPath ret = (EightyPath) hostFS.getPath( "" );
        for ( Path filename : guest ) {
            ret = (EightyPath) ret.resolve( filename.toString() );
        }

        return ret;
    }

    @Override
    public Optional<EightySymLink> getSymlink( EightyPath path ) {
        EightyPath hostLink = getHostPath( path );

        Optional<EightySymLink> hostSym = get80( hostLink ).getSymlink( hostLink );

        if ( !hostSym.isPresent() ) {
            return Optional.empty();
        }

        Path guestTarget = getGuestPath( path.getFileSystem(), hostSym.get().getTarget() );

        return Optional.of(new EightySymLink() {
            @Override
            public Path getTarget() {
                return guestTarget;
            }

            @Override
            public EightyPath getHost() {
                return path;
            }
        });

    }

    private Path getGuestPath( FileSystem fs, Path host ) {
        for ( String drive : drives.keySet() ) {
            if ( drives.get(drive).equals( host.getFileSystem())) {
                return  fs.getPath( drive + host.toString() );
            }
        }
        return host;
    }


    // ----------------------------------------------------
    private EightyPath getHostPath(EightyPath path) {
        String root = path.getRootComponent().orElse( "C:").toUpperCase(); // TODO for up UNC ?

        if ( root.startsWith("\\\\")) { // UNC
            String[] els = root.split("\\\\");
            String name = els[3];
            if ( name.length() != 2 || name.charAt(1) != '$' ) {
                throw runtime( new NoSuchFileException( "remote unc share not supported yet " + root ));
            }

            char d = name.charAt(0);

            if ( d < 'A' || 'Z' < d  ) {
                throw runtime( new NoSuchFileException( "unc share not supported yet " + root ));
            }

            root = "" + d + ":";
        }

        if ( !drives.containsKey(root)) {
            throw runtime( new NoSuchFileException( "not a valid drive " + root ));
        }

        EightyPath ret = (EightyPath) drives.get(root).getPath("/");
        EightyFS hostEighty = get80(ret);

        for ( Path elem : path ) {

            String elemStr = elem.toString();

            if ( Files.isDirectory( ret )) {// Files.exists( ret )) {
                ret = (EightyPath) hostEighty.newDirectoryStream(ret).
                        filter(kid -> nameEquals(elemStr, kid.getFileName().toString())).
                        findFirst().
                        orElse(ret.resolve(elemStr));
            } else {
                ret = (EightyPath) ret.resolve( elemStr );
            }

        }

        return ret;
    }

    private Path getGuestPath(Path guest, Path hostKid) {
        Path ret = guest.getRoot();

        for ( Path el : hostKid ) {
            ret = ret.resolve(el.toString());
        }

        return ret;
    }

}
