package de.pfabulist.lindwurm.eighty;

import de.pfabulist.kleinod.collection.Sets;
import de.pfabulist.kleinod.paths.Pathss;
import de.pfabulist.lindwurm.eighty.attributes.AttributeConnection;
import de.pfabulist.lindwurm.eighty.attributes.AttributeKeys;
import de.pfabulist.lindwurm.eighty.attributes.AttributesBuilder;
import de.pfabulist.lindwurm.eighty.attributes.LinkInfoSettable;
import de.pfabulist.lindwurm.eighty.path.EightyPath;
import de.pfabulist.lindwurm.eighty.path.URIMapper;

import java.io.IOException;
import java.net.URI;
import java.nio.channels.FileChannel;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.AccessMode;
import java.nio.file.ClosedFileSystemException;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.DirectoryStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileStore;
import java.nio.file.FileSystem;
import java.nio.file.FileSystemAlreadyExistsException;
import java.nio.file.FileSystemException;
import java.nio.file.FileSystemNotFoundException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.NotLinkException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.ProviderMismatchException;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributeView;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileAttributeView;
import java.nio.file.spi.FileSystemProvider;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import static de.pfabulist.kleinod.errors.Unchecked.*;
import static de.pfabulist.lindwurm.eighty.EightyUtils.get80;
import static de.pfabulist.lindwurm.eighty.attributes.AttributesBuilder.attributes;
import static java.nio.file.StandardCopyOption.COPY_ATTRIBUTES;
import static java.nio.file.StandardCopyOption.REPLACE_EXISTING;
import static java.nio.file.StandardOpenOption.APPEND;
import static java.nio.file.StandardOpenOption.CREATE;
import static java.nio.file.StandardOpenOption.CREATE_NEW;
import static java.nio.file.StandardOpenOption.READ;
import static java.nio.file.StandardOpenOption.WRITE;
import static java.util.Arrays.asList;

/**
 * ** BEGIN LICENSE BLOCK *****
 * BSD License (2 clause)
 * Copyright (c) 2006 - 2014, Stephan Pfab
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL Stephan Pfab BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * **** END LICENSE BLOCK ****
 */
public abstract class EightyProvider extends FileSystemProvider {

    private final ConcurrentMap<String, EightyFileSystem> fileSystems = new ConcurrentHashMap<>();

    private final URIMapper  uriMapper;
    private final EightyFSCreator creator;

    public EightyProvider( URIMapper uriMapper,
                           EightyFSCreator creator) {
        this.uriMapper = uriMapper;
        this.creator = creator;
    }


    @Override
    public String getScheme() {
        return uriMapper.getScheme();
    }

    @Override
    public FileSystem newFileSystem( URI uri, Map<String, ?> env2 ) throws IOException {

        checkURI( uri );

        Map<String,Object> env = (Map)env2;

        String id = uriMapper.getSchemeSpecificPart(uri);

        if ( id == null ) {
            throw new IllegalArgumentException( "scheme specific part is null : " + uri );
        }

        if ( fileSystems.containsKey( id )) {
            if ( fileSystems.get(id).isOpen()) {
                throw new FileSystemAlreadyExistsException( id );
            }
        }

        Object fsid = uriMapper.fromString(id, (Map<String,Object>)env);
        AttributesBuilder attributesBuilder = attributes();
        EightyFS efs = creator.create(fsid, attributesBuilder, (Map)env);

        if (efs == null ) {
            // when the baseFileSystemCreator is stubbed only
            return null;
        }

        EightyFileSystem eightyFileSystem = new EightyFileSystem( efs, id, this, attributesBuilder.build());
        fileSystems.put( id, eightyFileSystem );

        efs.setWatcher( eightyFileSystem );

        return eightyFileSystem;
    }

    @Override
    public FileSystem getFileSystem( URI uri ) {
        checkURI( uri );

        String id = uriMapper.getSchemeSpecificPart( uri );

        FileSystem ret = fileSystems.get( id );

        if ( ret == null ) {
            throw new FileSystemNotFoundException( uri.toString() );
        }

        if ( !ret.isOpen()) {
            fileSystems.remove(id); // for GC
            throw new FileSystemNotFoundException( uri.toString() );
        }


        return ret;
    }

    @Override
    public Path getPath( URI uri ) {
        checkURI( uri );
        FileSystem fs = Pathss.getOrCreate(uri, Collections.<String, Object>emptyMap());
        return fs.getPath( deUri( fs, uriMapper.getPathPart(uri)));
    }

    public EightyPath toRealPath( Path path ) {
        if ( !(path instanceof EightyPath )) {
            throw new IllegalArgumentException( path + " shouls be EightyPath" );
        }

        return (EightyPath) new RealPath( (EightyPath)path ).to();
    }

    public EightyPath toRealPathEx( Path path, LinkOption ... options  ) {

        if ( !(path instanceof EightyPath )) {
            throw new IllegalArgumentException( path + " shouls be EightyPath" );
        }

        if ( options.length == 0 ) {
            return toRealPath( path );
        }

        if ( path.getParent() == null ) {
            return (EightyPath)path;
        }
        
        EightyPath parent = toRealPath( path.getParent() );
        return (EightyPath) parent.resolve( path.getFileName());
    }

    @Override
    public SeekableByteChannel newByteChannel( Path pathArg, Set<? extends OpenOption> options, FileAttribute<?>... attrs ) throws IOException {

        final EightyFS eightyFS = checkProviderAndGet80( pathArg );
        final EightyFileSystem eightyFileSystem = (EightyFileSystem) pathArg.getFileSystem();

        EightyPath path = toRealPath( pathArg );

        allowAccess( path );

        Set<OpenOption> impliedOptions = new HashSet<>();
        impliedOptions.addAll( options );

        if ( impliedOptions.isEmpty() ) {
            impliedOptions.add( StandardOpenOption.READ );
        }

        if ( path.getParent() != null && !Files.exists(  path.getParent() )) {
            throw new NoSuchFileException( path.getParent().toString() );
        }

        if ( !Files.exists( path )) {
            if ( !options.contains( WRITE )) {
                throw new NoSuchFileException( path.toString() );
            }

            if ( !options.contains( CREATE ) && !options.contains( CREATE_NEW )) {
                throw new NoSuchFileException( path.toString() );
            }
        } else {

            if ( options.contains( CREATE_NEW )) {
                throw new FileAlreadyExistsException( path.toString() );
            }
        }

        if ( options.contains( APPEND ) && options.contains( READ )) {
            throw new IllegalArgumentException( "APPEND + READ not allowed"  );
        }

        if ( Files.isDirectory( path )) {
            throw new FileSystemException( path.toString() + " is a directory");
        }

        try {
            return eightyFileSystem.addClosable( eightyFS.newByteChannel( path, impliedOptions, attrs ));
        } catch( RuntimeException e ) {
            throw unwrap(IOException.class, e) ;
        }
    }

    @Override
    public FileChannel newFileChannel( Path path, Set<? extends OpenOption> options, FileAttribute<?>... attrs ) throws IOException {
        return super.newFileChannel( path, options, attrs );    //To change body of overridden methods use File | Settings | File Templates.
        // todo();
    }

    @Override
    public DirectoryStream<Path> newDirectoryStream( Path dirArg, DirectoryStream.Filter<? super Path> filter ) throws IOException {
        checkProviderAndGet80( dirArg );
        EightyFileSystem fs  = (EightyFileSystem) dirArg.getFileSystem();
        EightyPath       dir = toRealPath( dirArg);

        if ( !Files.exists( dir )) {
            throw new NoSuchFileException( "dir " + dir + " does not exist" );
        }

        // todo: test for throw

        try {
            return fs.addClosable( new EightyDirectoryStream(
                                            dirArg,
                                            EightyUtils.getDirectoryStream(dir),
                                            filter ));
        } catch( RuntimeException e ) {
            throw unwrap( IOException.class, e) ;
        }
    }

    @Override
    public void createDirectory( Path dirArg, FileAttribute<?>... attrs ) throws IOException {
        EightyFS   eighty = checkProviderAndGet80( dirArg );
        EightyPath dir    = toRealPathEx( dirArg, LinkOption.NOFOLLOW_LINKS);

        if ( existsEx( dir, LinkOption.NOFOLLOW_LINKS )) {
            throw new FileAlreadyExistsException( dir.toString() );
        }

        if ( !Files.isDirectory( dir.getParent() )) {
            throw new NoSuchFileException( dir.getParent().toString() );
        }

        allowAccess( dir );

        try {
            eighty.createDirectory( dir, attrs );
        } catch ( RuntimeException e ) {
            throw unwrap(IOException.class, e);
        }
    }


    @Override
    public void delete( Path pathArg ) throws IOException {
        EightyFS eightyFS = checkProviderAndGet80( pathArg );
        EightyPath path   = toRealPathEx( pathArg, LinkOption.NOFOLLOW_LINKS);

        Optional<EightySymLink> sym = eightyFS.getSymlink( path );

        if ( !Files.exists( path) && !sym.isPresent()) {
            throw new NoSuchFileException( "no such file" + pathArg );
        }

        if ( path.getParent() == null ) {
            throw new IllegalArgumentException( "can't delete root" );
        }

        if ( sym.isPresent() ) {
            eightyFS.delete( sym.get().getHost());
            return;
        }

        if ( Files.isDirectory( path) && !Pathss.isEmpty( path )) {
            throw new DirectoryNotEmptyException( "dir not empty" + pathArg );
        }

        // todo: write a test for throw

        try {
            eightyFS.delete( path );
        } catch (RuntimeException e) {
            throw unwrap( IOException.class, e );
        }
    }

    @Override
    public void copy( Path osource, Path otarget, CopyOption... options ) throws IOException {
        checkProviderAndGet80( otarget );
        EightyFS srcEightyFS = checkProviderAndGet80( osource );

        if ( Files.isSameFile( osource, otarget )) {
            // no op
            return;
        }

//        EightyPath target = normPath(otarget);
//        EightyPath source = normPath(osource);

        if ( !asList( options ).contains( REPLACE_EXISTING )) {
            if ( existsEx( otarget, LinkOption.NOFOLLOW_LINKS )) {
                throw new FileAlreadyExistsException( otarget.toString() );
            }
        }

        if ( Files.isDirectory( otarget )) {
            if ( !Pathss.isEmpty( otarget )) {
                throw new DirectoryNotEmptyException("" + otarget + " is not empty");
            }
        }

        if ( existsEx( otarget, LinkOption.NOFOLLOW_LINKS )) {
            Files.delete( otarget);
        }

        EightyPath source = toRealPath( osource );
        EightyPath target = toRealPath( otarget );

        if ( Files.isDirectory(source)) {
            Files.createDirectory( target );
            return;
        }

        try {
            srcEightyFS.copy( source, target, options );
        } catch( RuntimeException e ) {
            throw unwrap( IOException.class, e );
        }

        if ( asList( options ).contains( COPY_ATTRIBUTES )) { // todo link: where to set
            Files.setLastModifiedTime( target, Files.getLastModifiedTime(source));
        }
    }

    @Override
    public void move( Path osource, Path otarget, CopyOption... options ) throws IOException {
        EightyFS source80 = checkProviderAndGet80( osource );
        EightyFS target80 = checkProviderAndGet80( otarget );

        EightyPath target = toRealPathEx( otarget, LinkOption.NOFOLLOW_LINKS);
        EightyPath source = toRealPathEx( osource, LinkOption.NOFOLLOW_LINKS );

        if ( source.getParent() == null ) {
            throw new FileSystemException( "root can not be moved" );
        }

        if ( !asList( options ).contains( REPLACE_EXISTING )  && Files.exists( target, LinkOption.NOFOLLOW_LINKS )) {
            throw new FileAlreadyExistsException( target.toString() );
        }

        if ( Files.isDirectory( target ) && !Pathss.isEmpty( target )) {
            throw new DirectoryNotEmptyException(target.toString());
        }

        if ( !Files.exists( target.getParent())) {
            throw new NoSuchFileException( "parent of target is missing " + otarget );
        }

        allowAccess( target );

        Files.deleteIfExists( target ); // todo if target is sym link ?

        // todo target link
        // todo: ! hard link dirs can make this interesting


        Path parent = target.getParent();

        while ( parent != null ) {
            if ( parent.equals( source )) {
                throw new FileSystemException( "a directory can not be moved into one of its own subdirectories" );
            }
            parent = parent.getParent();
        }


        if ( Files.isSymbolicLink( source )) {
            Files.createSymbolicLink( target, source80.getSymlink( source ).get().getTarget());
//            Files.setLastModifiedTime( target, Files.getLastModifiedTime( source, LinkOption.NOFOLLOW_LINKS ));
            Files.delete( source );
            return;
        }

        try {
            source80.move( source, target, options);
        } catch( RuntimeException e ) {
            throw unwrap( IOException.class, e );
        }
    }

    @Override
    public boolean isSameFile( Path opath, Path opath2 ) throws IOException {

        if ( !opath.getFileSystem().provider().equals( opath2.getFileSystem().provider() )) {
            return false;
        }

        checkProviderAndGet80( opath );

        if ( opath.equals( opath2 )) {
            return true;
        }

        if ( !Files.exists( opath )) {
            if ( Files.exists(opath2)) {
                return false;
            }
        }

        if ( !Files.exists( opath2 )) {
            if ( Files.exists(opath)) {
                return false;
            }

            Path r1 = ((EightyProvider)opath.getFileSystem().provider()).toRealPathEx( opath, LinkOption.NOFOLLOW_LINKS );
            Path r2 = ((EightyProvider)opath.getFileSystem().provider()).toRealPathEx( opath2, LinkOption.NOFOLLOW_LINKS );

            return r1.equals( r2 );
        }

        // eightyfilesystems supporting hardlinks must support filekey
        Object fk1 = Files.readAttributes( opath,  BasicFileAttributes.class).fileKey();
        Object fk2 = Files.readAttributes( opath2, BasicFileAttributes.class).fileKey();

        if ( fk1 != null && fk2 != null && fk1.equals( fk2 )) {
            return true;
        }

        Path r1 = opath.toRealPath();
        Path r2 = opath2.toRealPath();

        // if there no hardlinks to dirs, path to file is now unique
        return r1.equals( r2 );

////        if ( isOtherFileSystem( r1, path)) {
////            return Files.isSameFile( r1, r2 );
////        }
//
//        if ( !r1.equals( opath ) || !r2.equals( opath2 )) {
//            return isSameFile( r1, r2 );
//        }
//
//        if ( r1.equals( r2 )) {
//            return true;
//        }
//
//        return false;
    }

    @Override
    public boolean isHidden( Path path ) throws IOException {
        checkProviderAndGet80( path );

        if ( !Files.exists( path )) {
            throw new NoSuchFileException( path.toString() );
        }

        return checkProviderAndGet80( path ).isHidden( toRealPath( path ) );
    }

    @Override
    public FileStore getFileStore( Path path ) throws IOException {
        checkProviderAndGet80( path );

        if ( !Files.exists( path )) {
            throw new NoSuchFileException( path.toString() );
        }

        return checkProviderAndGet80( path ).getFileStore( toRealPath( path ));
    }

    @Override
    public void checkAccess( Path pathArg, AccessMode... modes ) throws IOException {

        EightyFS   efs  = checkProviderAndGet80( pathArg );
        EightyPath path = toRealPath( pathArg);

        try {
            efs.checkAccess( path, modes );
        } catch ( RuntimeException e) {
            throw unwrap( IOException.class, e );
        }
    }

    @Override
    public <V extends FileAttributeView> V getFileAttributeView( Path path, final Class<V> type, LinkOption... options ) {
        EightyFS efs = checkProviderAndGet80( path );

        if ( !existsEx( path, options )) {

            if ( !((EightyFileSystem)path.getFileSystem()).getAttributeProvider().supportsView(type)) {
                return null;
            }

            return RuntimeProxy.of( type, name -> {
                    if ( name.equals("name") ) {
                        return "FileAttributeProxy";
                    }

                    throw new NoSuchFileException( path.toString());
                });
        }

        EightyPath real = toRealPathEx( path, options );

        if ( !((EightyFileSystem)path.getFileSystem()).getAttributeProvider().supportsView(type)) {
            return null;
        }

        Optional<EightySymLink> sym = efs.getSymlink( real);
        if ( sym.isPresent()) {
            V fav =  efs.getFileAttributeView( sym.get().getHost(), type );

            if ( !(fav instanceof LinkInfoSettable )) {
                throw new UnsupportedOperationException( "the attribute view need to implement LinkInfoSettable in order to make SymLinks work" );
            }

            ((LinkInfoSettable)(fav)).setLink();

            return fav;
        }

        return efs.getFileAttributeView( real, type );
    }

    /** same as Files.exists( NOFOLLOWLINKS )
     * but without checking attributes
     *
     */
    private boolean existsEx( Path path, LinkOption ... options ) {
        if ( Files.exists( path )) {
            return true;
        }

        if ( options.length == 0 || !options[0].equals( LinkOption.NOFOLLOW_LINKS )) {
            return false;
        }


        try {
            Files.readSymbolicLink( path );
        } catch( IOException e ) {
            return false;
        }

        return true;
    }

    @Override
    public <A extends BasicFileAttributes> A readAttributes( Path path, final Class<A> type, LinkOption... options ) throws IOException {
        checkProviderAndGet80(path);
//        Path normPath = normPath( path );

        if ( !existsEx( path, options )) {
            throw new NoSuchFileException( path + " does not exist" );
        }

        // todo test custom atti, should fail if target fs does not have it

        Class<? extends BasicFileAttributeView> view =
                ((EightyFileSystem)path.getFileSystem()).
                        getAttributeProvider().
                        getViewfromRead( type ).orElseThrow(() -> new UnsupportedOperationException( type + " not a supported FileAttributes class" ));

        return (A) getFileAttributeView( path, view, options ).readAttributes();

    }


    @Override
    public Map<String, Object> readAttributes( Path path, String attributes, LinkOption... options ) throws IOException {
        EightyFS efs = checkProviderAndGet80( path );

        if ( !existsEx( path, options )) {
            throw new NoSuchFileException( path + " does not exist" );
        }

        EightyPath real = toRealPathEx( path, options );

        final String viewName = AttributeKeys.getName( attributes );

        AttributeConnection<BasicFileAttributeView, BasicFileAttributes> connection =
                real.getFileSystem().getAttributeProvider().
                        getConnectionFromName(viewName).orElseThrow(() -> new UnsupportedOperationException(viewName + " is not a supported FileAttributeView"));

        BasicFileAttributeView view = getFileAttributeView( real, connection.getViewType(), options);
        Map<String, Object>    ret  = new HashMap<>();

        for ( String key : AttributeKeys.getKeys(attributes, connection.getAttributeNames()) ) {
            ret.put( key, connection.get( view.readAttributes(), key ));
        }

        return ret;
    }

    @Override
    public void setAttribute( Path path, final String attribute, Object value, LinkOption... options ) throws IOException {
        EightyFS efs = checkProviderAndGet80( path );

        // todo: link test // exiss test

        final String viewName = AttributeKeys.getName(attribute);

        AttributeConnection<BasicFileAttributeView, BasicFileAttributes> connection =
                ((EightyFileSystem)path.getFileSystem()).getAttributeProvider().
                        getConnectionFromName(viewName).orElseThrow(() -> new UnsupportedOperationException(viewName + " is not a supported FileAttributeView"));

        Set<String> keys = AttributeKeys.getKeys(attribute, connection.getAttributeNames());

        if ( keys.size() != 1 ) {
            throw new IllegalArgumentException( "you can set only one attribute at a time" );
        }


        BasicFileAttributeView view = getFileAttributeView( path, connection.getViewType(), options );

        connection.set( view, keys.iterator().next(), value );
    }

    @Override
    public void createLink( Path link, Path existing) throws IOException {

        EightyFS   efs          = checkProviderAndGet80( existing );
        checkProviderAndGet80( link );
//        EightyPath normExisting = normPath( existing );
//        EightyPath normLink     = normPath( link );

        if ( existsEx( link, LinkOption.NOFOLLOW_LINKS )) {
            throw new FileAlreadyExistsException( link.toString() );
        }

        if ( !link.getFileSystem().equals( existing.getFileSystem()) ) {
            throw new FileSystemException( "cannot link to a different filesystem" );
        }

        EightyPath rLink = toRealPath( link );

        allowAccess( rLink);

        EightyPath rExisting = toRealPath( existing );

        try {
            efs.createHardLink( rLink, rExisting );
        } catch ( RuntimeException e ) {
            throw unwrap( IOException.class, e );
        }

    }

    @Override
    public void createSymbolicLink( Path link, Path target, FileAttribute<?>... attrs) throws IOException {

        EightyFS   efs      = checkProviderAndGet80( link );
        checkProviderAndGet80( target );

        EightyPath normLink = toRealPath( link );

        if ( Files.exists( normLink )) {
            throw new FileAlreadyExistsException( link.toString() );
        }

        allowAccess( normLink );

        efs.createSymLink( normLink, (EightyPath)target, attrs);
    }

    @Override
    public Path readSymbolicLink( Path link ) throws IOException {
        EightyFS   efs      = checkProviderAndGet80( link );
        EightyPath real     = toRealPathEx( link, LinkOption.NOFOLLOW_LINKS );

        // todo: no such file

        Optional<EightySymLink> sym = efs.getSymlink( real);

        if ( sym.isPresent() ) {
            return sym.get().getTarget();
        }

        throw new NotLinkException( link.toString() );
    }

    public URIMapper getUriMapper() {
        return uriMapper;
    }

    /*
     * -------------------------------------- private --------------------------------------------------------------
     */

    private boolean isOtherFileSystem( Path nn, EightyPath old ) {
        return !nn.getFileSystem().equals( old.getFileSystem());
    }

    private void allowAccess( EightyPath path ) throws FileSystemException {
        Optional<String> message = get80(path).allowAccess( path );

        if ( message.isPresent() ) {
            throw new FileSystemException( message.get() );
        }
    }

    private EightyFS checkProviderAndGet80( Path path ) {
        if ( path.getFileSystem().provider() != this ) {
            throw new ProviderMismatchException( "expected path for " + getScheme() + " got " + path.getFileSystem().provider().getScheme() );
        }

        if ( !path.getFileSystem().isOpen()) {
            throw new ClosedFileSystemException();
        }

        return ( (EightyFileSystem) path.getFileSystem() ).get80();
    }

    private void checkURI( URI uri ) {
        if ( !uri.getScheme().equals( getScheme() )) {
            throw new IllegalArgumentException( "scheme does not fit filesystem, '"+ getScheme() +"' expected got " + uri.getScheme() );
        }
    }

    private String deUri( FileSystem fs, String path) {
        return path.replace( "/",  fs.getSeparator());
    }



    // todo useage ?
    public void removeAllFileSystemsBut( EightyFileSystem ro ) {
        String key= null;
        for ( Map.Entry<String, EightyFileSystem> pair : fileSystems.entrySet() ) {
            if ( pair.getValue().equals( ro )) {
                key = pair.getKey();
            }
        }

        fileSystems.clear();
        if ( key != null ) {
            fileSystems.put( key, ro );
        }
    }

}
