package de.pfabulist.lindwurm.eighty;

import de.pfabulist.kleinod.collection.P;

import java.nio.file.InvalidPathException;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * ** BEGIN LICENSE BLOCK *****
 * BSD License (2 clause)
 * Copyright (c) 2006 - 2014, Stephan Pfab
 * All rights reserved.
 * <p>
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * <p>
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL Stephan Pfab BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * **** END LICENSE BLOCK ****
 */
public class WindowsPathConstraints implements EightyPathConstraints {
    @Override
    public boolean nameEquals(String a, String b) {
        return a.equalsIgnoreCase( b );
    }

    @Override
    public int nameHash(String a) {
        return a.toUpperCase().hashCode();
    }

    public String getSeparators() {
        return "\\/";
    }

    @Override
    public P<Optional<String>, String> parseRootComponent(String str) {

        if ( str.startsWith("\\\\") || str.startsWith( "//" ) || str.startsWith("\\/") || str.startsWith("/\\")) {
            String[] elms = str.split( "\\\\|/");

            List<String> elems = Arrays.asList(elms).stream().
                    filter( x-> !x.isEmpty()).
                    collect(Collectors.toList());

            if ( elems.size() < 2 ) {
                throw new InvalidPathException( str, "not a valid unc path " );
            }

            String norm            = elems.stream().reduce("\\", (before, elem ) -> before + "\\" + elem );
            if ( elems.size() == 2 ) {
                norm += "\\";
            }
            int    rootCompoLength = 3 + elems.get(0).length() + elems.get(1).length();

            return P.of( Optional.of( norm.substring( 0, rootCompoLength )), norm.substring( rootCompoLength ));
        }

        if ( str.length() < 2 || str.charAt(1) != ':' ) {
            return P.of(Optional.<String>empty(), str );
        }

        char ch = str.toUpperCase().charAt(0);

        if ( ch >= 'A' && ch <= 'Z') {
            return P.of(Optional.of( "" + ch + ":"), str.substring(2) );
        }

        return P.of(Optional.<String>empty(), str);
    }

    @Override
    public Optional<String> allowAccess( EightyPath path) {
        //EightyFS eightyFS = EightyUtils.get80(path);

        Optional<String> message = EightyUtils.checkFilenameLength(path, 255);

        if ( message.isPresent() ) {
            return message;
        }

        for ( Path filename : path ) {
            if ( filename.toString().equalsIgnoreCase("nul")) {
                return Optional.of("illegal filename: nul");
            }
        }

        return Optional.empty();
    }

}
