package de.pfabulist.lindwurm.eighty;

import de.pfabulist.kleinod.paths.PathUtils;

import java.io.IOException;
import java.net.URI;
import java.nio.channels.FileChannel;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.*;
import java.nio.file.attribute.BasicFileAttributeView;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileAttributeView;
import java.nio.file.spi.FileSystemProvider;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import static de.pfabulist.lindwurm.eighty.AttributesBuilder.attributes;
import static java.nio.file.StandardCopyOption.COPY_ATTRIBUTES;
import static java.nio.file.StandardCopyOption.REPLACE_EXISTING;
import static java.nio.file.StandardOpenOption.APPEND;
import static java.nio.file.StandardOpenOption.CREATE;
import static java.nio.file.StandardOpenOption.CREATE_NEW;
import static java.nio.file.StandardOpenOption.READ;
import static java.nio.file.StandardOpenOption.WRITE;

/**
 * ** BEGIN LICENSE BLOCK *****
 * BSD License (2 clause)
 * Copyright (c) 2006 - 2014, Stephan Pfab
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL Stephan Pfab BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * **** END LICENSE BLOCK ****
 */
public abstract class EightyProvider extends FileSystemProvider {

    private final ConcurrentMap<String, EightyFileSystem> fileSystems = new ConcurrentHashMap<>();

    private final URIMapper  uriMapper;
//    private final BiFunction<Object, Map<String, Object>, EightyFS> baseFileSystemCreator;
//
//    final AttributesProvider attributeProvider;
    private final EightyFSCreator creator;

    public EightyProvider( URIMapper uriMapper,
                           EightyFSCreator creator
//                           BiFunction<Object, Map<String, Object>, EightyFS> baseFileSystemCreator,
//                           AttributesProvider attributeProvider
    ) {
        this.uriMapper = uriMapper;
        this.creator = creator;
//        this.baseFileSystemCreator = baseFileSystemCreator;
//        this.attributeProvider = attributeProvider;
    }


    @Override
    public String getScheme() {
        return uriMapper.getScheme();
    }

    @Override
    public FileSystem newFileSystem( URI uri, Map<String, ?> env2 ) throws IOException {

        checkURI( uri );

        Map<String,Object> env = (Map<String, Object>) env2;

        String id = uriMapper.getSchemeSpecificPart( uri );

        if ( id == null ) {
            throw new IllegalArgumentException( "scheme specific part is null : " + uri );
        }

        if ( fileSystems.containsKey( id )) {
            if ( fileSystems.get(id).isOpen()) {
                throw new FileSystemAlreadyExistsException( id );
            }
        }

        Object fsid = uriMapper.fromString(id, (Map<String,Object>)env);
        AttributesBuilder attributesBuilder = attributes();
        EightyFS efs = creator.create(fsid, attributesBuilder, (Map)env);

        if (efs == null ) {
            // when the baseFileSystemCreator is stubbed only
            return null;
        }

        EightyFileSystem eightyFileSystem = new EightyFileSystem( efs, id, this, attributesBuilder.build());
        fileSystems.put( id, eightyFileSystem );

        efs.setWatcher( eightyFileSystem );

        return eightyFileSystem;
    }

    @Override
    public FileSystem getFileSystem( URI uri ) {
        checkURI( uri );

        String id = uriMapper.getSchemeSpecificPart( uri );

        FileSystem ret = fileSystems.get( id );

        if ( ret == null ) {
            throw new FileSystemNotFoundException( uri.toString() );
        }

        if ( !ret.isOpen()) {
            fileSystems.remove(id); // for GC
            throw new FileSystemNotFoundException( uri.toString() );
        }


        return ret;
    }

    @Override
    public Path getPath( URI uri ) {
        checkURI( uri );

        try {
            FileSystem fs = PathUtils.getOrCreate( uri, Collections.<String, Object>emptyMap() );
            return fs.getPath( deUri( fs, uriMapper.getPathPart(uri)));
        } catch( IOException e ) {
            throw new FileSystemNotFoundException( uri.toString() );
        }

    }

    private String deUri( FileSystem fs, String path) {
        return path.replace( "/",  fs.getSeparator());
    }

    @Override
    public SeekableByteChannel newByteChannel( Path path, Set<? extends OpenOption> options, FileAttribute<?>... attrs ) throws IOException {

        final EightyFS eightyFS = checkProviderAndGet80( path );
        final EightyFileSystem eightyFileSystem = (EightyFileSystem) path.getFileSystem();
        EightyPath normPath = normPath( path );

        allowAccess( normPath );

        Set<OpenOption> impliedOptions = new HashSet<>();
        impliedOptions.addAll( options );

        if ( impliedOptions.isEmpty() ) {
            impliedOptions.add( StandardOpenOption.READ );
        }

        if ( normPath.getParent() != null && !Files.exists( normPath.getParent() )) {
            throw new NoSuchFileException( normPath.getParent().toString() );
        }

        if ( !Files.exists( normPath )) {
            if ( !options.contains( WRITE )) {
                throw new NoSuchFileException( normPath.toString() );
            }

            if ( !options.contains( CREATE ) && !options.contains( CREATE_NEW )) {
                throw new NoSuchFileException( normPath.toString() );
            }
        } else {

            if ( options.contains( CREATE_NEW )) {
                throw new FileAlreadyExistsException( normPath.toString() );
            }
        }

        if ( options.contains( APPEND ) && options.contains( READ )) {
            throw new IllegalArgumentException( "APPEND + READ not allowed"  );
        }

        EightyPath cleanPath = (EightyPath) normPath.toAbsolutePath();

        if ( Files.isDirectory( cleanPath )) {
            throw new FileSystemException( cleanPath.toString() + " is a directory");
        }

        return eightyFileSystem.addClosable(eightyFS.newByteChannel(cleanPath, impliedOptions, attrs));
    }

    @Override
    public FileChannel newFileChannel( Path path, Set<? extends OpenOption> options, FileAttribute<?>... attrs ) throws IOException {
        return super.newFileChannel( path, options, attrs );    //To change body of overridden methods use File | Settings | File Templates.
        // todo();
    }

    @Override
    public DirectoryStream<Path> newDirectoryStream( Path dir, DirectoryStream.Filter<? super Path> filter ) throws IOException {
        EightyFS efs = checkProviderAndGet80( dir );
        EightyFileSystem fs = (EightyFileSystem) dir.getFileSystem();

        if ( !Files.exists( dir )) {
            throw new NoSuchFileException( "dir " + dir + " does not exist" );
        }

        return fs.addClosable( new EightyDirectoryStream(
                                        dir,
                                        efs.newDirectoryStream( normPath( dir ) ),
                                        filter ));
    }

    @Override
    public void createDirectory( Path dir, FileAttribute<?>... attrs ) throws IOException {
        EightyFS eighty = checkProviderAndGet80( dir );

        if ( dir.getParent() == null ) {
            // root
            throw new FileAlreadyExistsException( dir.toString() );
        }

        if ( !Files.isDirectory( dir.getParent() )) {
            throw new NoSuchFileException( dir.getParent().toString() );
        }

        if ( Files.isDirectory( dir ) || Files.isRegularFile( dir )) {    // TODO symlink
            throw new FileAlreadyExistsException( dir.toString() );
        }

        EightyPath normDir = normPath(dir);
        allowAccess( normDir );

        eighty.createDirectory( normDir, attrs );

    }

    @Override
    public void delete( Path path ) throws IOException {
        EightyFS eightyFS = checkProviderAndGet80( path );
        EightyPath nPath = normPath(path);

        if ( !Files.exists(nPath)) {
            throw new NoSuchFileException( "no such file" + path );
        }

        if ( nPath.getParent() == null ) {
            throw new IllegalArgumentException( "can't delete root" );
        }

        if ( Files.isDirectory(nPath) && !PathUtils.isEmpty( nPath )) {
            throw new DirectoryNotEmptyException( "dir not empty" + path );
        }

        eightyFS.delete(nPath);
    }

    @Override
    public void copy( Path osource, Path otarget, CopyOption... options ) throws IOException {
        checkProviderAndGet80( otarget );
        EightyFS srcEightyFS = checkProviderAndGet80( osource );

        EightyPath target = normPath(otarget);
        EightyPath source = normPath(osource);

        if ( !Arrays.asList(options).contains( REPLACE_EXISTING )) {
            if ( Files.exists(target)) {
                throw new FileAlreadyExistsException( otarget.toString() );
            }
        }

        if ( Files.isDirectory( target )) {
            if ( !PathUtils.isEmpty( target )) {
                throw new DirectoryNotEmptyException("" + target + " is not empty");
            }
        }

        if ( Files.isDirectory(target) || Files.isRegularFile(target)) {
            Files.delete(target);
        }

        if ( Files.isDirectory(source)) {
            Files.createDirectory( target );
            return;
        }


        srcEightyFS.copy( normPath(osource), target, options );

        if ( Arrays.asList( options ).contains( COPY_ATTRIBUTES )) {
            Files.setLastModifiedTime(target, Files.getLastModifiedTime(source));
        }
    }

    @Override
    public void move( Path osource, Path otarget, CopyOption... options ) throws IOException {
        EightyFS source80 = checkProviderAndGet80( osource );
        EightyFS target80 = checkProviderAndGet80( otarget );

        EightyPath target = normPath(otarget);
        EightyPath source = normPath(osource);

        allowAccess( target );

        if ( source.getParent() == null ) {
            throw new FileSystemException( "root can not be moved" );
        }

        Path parent = target.getParent();

        while ( parent != null ) {
            if ( parent.equals(source)) {
                throw new FileSystemException( "a directory can not be moved into one of its own subdirectories" );
            }
            parent = parent.getParent();
        }

        if ( !Arrays.asList( options ).contains( REPLACE_EXISTING )
                && (Files.isRegularFile(target) || Files.isDirectory(target))) {
            throw new FileAlreadyExistsException( target.toString() );
        }

        if ( Files.isDirectory(target) && !PathUtils.isEmpty( target)) {
            throw new DirectoryNotEmptyException(target.toString());
        }

        if ( !Files.exists( target.getParent())) {
            throw new NoSuchFileException( "parent of target is missing " + otarget );
        }

        Files.deleteIfExists( target );

        source80.move( source, target, options );
    }

    @Override
    public boolean isSameFile( Path opath, Path opath2 ) throws IOException {

        if ( !opath.getFileSystem().provider().equals( opath2.getFileSystem().provider() )) {
            return false;
        }

        checkProviderAndGet80( opath );

        Path path = normPath(opath);
        Path path2 = normPath(opath2);

        if ( path.equals( path2 )) {
            return true;
        }

        if ( !Files.exists( path )) {
            throw new NoSuchFileException( path.toString() );
        }

        if ( !Files.exists(path2)) {
            throw new NoSuchFileException( path2.toString() );
        }

        Path r1 = path.toRealPath();
        Path r2 = path2.toRealPath();

        if ( !r1.equals( path) || !r2.equals(path2)) {
            return isSameFile( r1, r2 );
        }

        return checkProviderAndGet80( path ).isSameFile( normPath(path), normPath(path2));
    }

    @Override
    public boolean isHidden( Path path ) throws IOException {
        return checkProviderAndGet80( path ).isHidden( normPath( path ));
    }

    @Override
    public FileStore getFileStore( Path path ) throws IOException {
        return checkProviderAndGet80( path ).getFileStore( normPath( path ));
    }

    @Override
    public void checkAccess( Path path, AccessMode... modes ) throws IOException {
        checkProviderAndGet80( path ).checkAccess( normPath(path), modes );
    }

    @Override
    public <V extends FileAttributeView> V getFileAttributeView( Path path, final Class<V> type, LinkOption... options ) {
        EightyFS efs = checkProviderAndGet80( path );

        if ( !((EightyFileSystem)path.getFileSystem()).getAttributeProvider().supportsView(type)) {
            return null;
        }

        EightyPath normPath = normPath( path );

        if ( !Files.exists( normPath )) {
            return RuntimeProxy.of( type, name -> {
                                                    if ( name.equals("name") ) {
                                                        return "FileAttributeProxy";
                                                    }

                                                    throw new NoSuchFileException( path.toString());
                                                });
        }

        return efs.getFileAttributeView( normPath, type, options );
    }

    @Override
    public <A extends BasicFileAttributes> A readAttributes( Path path, final Class<A> type, LinkOption... options ) throws IOException {
        EightyFS efs = checkProviderAndGet80( path );
        Path normPath = normPath( path );

        if ( !Files.exists( normPath )) {
            throw new NoSuchFileException( normPath + " does not exist" );
        }

        Class<? extends BasicFileAttributeView> view = ((EightyFileSystem)path.getFileSystem()).getAttributeProvider().
                getViewfromRead(type).orElseThrow(() -> new UnsupportedOperationException( type + " not a supported FileAttributes class" ));
        return (A) getFileAttributeView( normPath, view, options ).readAttributes();

//        AttributeInfo info = efs.supportedAttributes().stream().
//                filter( ainfo -> ainfo.attributes.equals(type)).
//                findFirst().
//                orElseThrow( () -> new UnsupportedOperationException( type + " not a supported FileAttributes class" ));
//
//        return (A) getFileAttributeView( normPath, info.view, options ).readAttributes();
    }


    @Override
    public Map<String, Object> readAttributes( Path path, String attributes, LinkOption... options ) throws IOException {
        EightyFS efs = checkProviderAndGet80( path );

        if ( !Files.exists( path )) {
            throw new NoSuchFileException( path + " does not exist" );
        }

        final String viewName = AttributeKeys.getName( attributes );

        AttributeConnection<BasicFileAttributeView, BasicFileAttributes> connection =
                ((EightyFileSystem)path.getFileSystem()).getAttributeProvider().
                        getConnectionFromName(viewName).orElseThrow(() -> new UnsupportedOperationException(viewName + " is not a supported FileAttributeView"));

        BasicFileAttributeView view = getFileAttributeView(path, connection.getViewType(), options);
        Map<String, Object>    ret  = new HashMap<>();

        for ( String key : AttributeKeys.getKeys(attributes, connection.getAttributeNames()) ) {
            ret.put( key, connection.get( view.readAttributes(), key ));
        }

        return ret;

//        AttributeInfo info = efs.supportedAttributes().stream().
//                filter( ainfo -> ainfo.name.equals(viewName)).
//                findFirst().
//                orElseThrow(
//
//
//                        BasicFileAttributeView view = getFileAttributeView(path, info.view, options);
//
//        Map<String, Object> ret = new HashMap<>();
//
//        for ( String key : AttributeKeys.getKeys( attributes, info.getSet.getAttributeNames() ) ) {
//            ret.put( key, info.getSet.get( view, key ));
//        }
//
//        return ret;
    }

    @Override
    public void setAttribute( Path path, final String attribute, Object value, LinkOption... options ) throws IOException {
        EightyFS efs = checkProviderAndGet80( path );

        final String viewName = AttributeKeys.getName(attribute);

        AttributeConnection<BasicFileAttributeView, BasicFileAttributes> connection =
                ((EightyFileSystem)path.getFileSystem()).getAttributeProvider().
                        getConnectionFromName(viewName).orElseThrow(() -> new UnsupportedOperationException(viewName + " is not a supported FileAttributeView"));

        Set<String> keys = AttributeKeys.getKeys( attribute, connection.getAttributeNames() );

        if ( keys.size() != 1 ) {
            throw new IllegalArgumentException( "you can set only one attribute at a time" );
        }


        BasicFileAttributeView view = getFileAttributeView( path, connection.getViewType(), options );

        connection.set( view, keys.iterator().next(), value );
//
//        AttributeInfo info = efs.supportedAttributes().stream().
//                filter( ainfo -> ainfo.name.equals( viewName )).
//                findFirst().
//                orElseThrow( () -> new UnsupportedOperationException( viewName + " is not a supported FileAttributeView" ));
//
//        Set<String> keys = AttributeKeys.getKeys( attribute, info.getSet.getAttributeNames() );
//
//        if ( keys.size() != 1 ) {
//            throw new IllegalArgumentException( "you can set only one attribute at a time" );
//        }
//
//
//        BasicFileAttributeView view = getFileAttributeView( path, info.view, options );
//
//        info.getSet.set( view, keys.iterator().next(), value );
    }

    @Override
    public void createLink(Path link, Path existing) throws IOException {

        EightyFS   efs          = checkProviderAndGet80( existing );
        EightyPath normExisting = normPath( existing );
        EightyPath normLink     = normPath( link );

        if ( Files.exists( normLink )) {
            throw new FileAlreadyExistsException( link.toString() );
        }

        if ( !link.getFileSystem().equals( existing.getFileSystem()) ) {
            throw new FileSystemException( "cannot link to a different filesystem" );
        }

        allowAccess( normLink );

        efs.createHardLink( normLink, normExisting );

    }


    /*
     * -------------------------------------- private --------------------------------------------------------------
     */

    private void allowAccess( EightyPath path ) throws FileSystemException {
        Optional<String> message = EightyUtils.get80(path).allowAccess( path );

        if ( message.isPresent() ) {
            throw new FileSystemException( message.get() );
        }
    }

    private EightyFS checkProviderAndGet80( Path path ) {
        if ( path.getFileSystem().provider() != this ) {
            throw new ProviderMismatchException( "expected path for " + getScheme() + " got " + path.getFileSystem().provider().getScheme() );
        }

        if ( !path.getFileSystem().isOpen()) {
            throw new ClosedFileSystemException();
        }

        return ( (EightyFileSystem) path.getFileSystem() ).get80();
    }

    private EightyPath normPath( Path path ) {
        return (EightyPath)(path.normalize().toAbsolutePath());
    }

    private void checkURI( URI uri ) {
        if ( !uri.getScheme().equals( getScheme() )) {
            throw new IllegalArgumentException( "scheme does not fit filesystem, '"+ getScheme() +"' expected got " + uri.getScheme() );
        }
    }

    // todo useage ?
    public void removeAllFileSystemsBut( EightyFileSystem ro ) {
        String key= null;
        for ( Map.Entry<String, EightyFileSystem> pair : fileSystems.entrySet() ) {
            if ( pair.getValue().equals( ro )) {
                key = pair.getKey();
            }
        }

        fileSystems.clear();
        if ( key != null ) {
            fileSystems.put( key, ro );
        }
    }

    public URIMapper getUriMapper() {
        return uriMapper;
    }
}
