package de.pfabulist.lindwurm.eighty;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.*;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileAttributeView;
import java.util.*;
import java.util.stream.Stream;

/**
 * ** BEGIN LICENSE BLOCK *****
 * BSD License (2 clause)
 * Copyright (c) 2006 - 2014, Stephan Pfab
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL Stephan Pfab BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * **** END LICENSE BLOCK ****
 */

/**
 * EightFs is an interface that helps implement NIO2 filesystems
 *
 * the methods can assume that all Path arguments are from this filesystem and absolute
 */
public interface EightyFS extends EightyPathConstraints {

    /**
     * Creates new CloseableSeekableByteChannel to that path
     * This method works in mostly the manner
     * specified by the {@link java.nio.file.Files#newByteChannel} method.
     *
     * Differences to the general method:
     * - The set of options if not empty
     * - APPEND and READ are not both in the options
     * - If READ is in the options than the file exists
     * - IF the file does not exist than WRITE and either CREATE or CREATE_NEW are in the options
     * - The path is not a directory
     * - The path is absolute
     * - The channel needs to implement InformativeClosable
     *
     * @param path An absolute path of a filesystem of this provider
     * @param options A non empty set of options without any conflicts, see above
     * @param attrs attributes
     * @return
     * @throws IOException
     *
     *
     */
    CloseableSeekableByteChannel newByteChannel( EightyPath path, Set<? extends OpenOption> options, FileAttribute<?>... attrs ) throws IOException;

    /**
     * Create a new Directory
     * This method should work in the manner specified in the {@link java.nio.file.Files#createDirectory} method.
     *
     * Differences are
     * - dir is absolute
     * - dirs parent is a dir and exists
     * - dir does not exist
     *
     * @param dir
     * @param attrs
     * @throws IOException
     */
    void createDirectory( EightyPath dir, FileAttribute<?>... attrs ) throws IOException;

    /**
     * Check Read, Write, Exec access
     * is called quite often for file existence
     */
    void checkAccess( EightyPath path, AccessMode... modes ) throws IOException;

    Stream<Path> newDirectoryStream( EightyPath dir );

    /**
     * Copy with same provider (not necessarily same FS)
     * target does not exist
     * needs to send out watch events (if supported in FS)
     * @param source
     * @param target
     * @param options
     * @throws IOException
     */
    default void copy( EightyPath source, EightyPath target, CopyOption... options ) throws IOException {
        try (InputStream in = Files.newInputStream(source)) {
            Files.copy(in, target);
        }
    }

    /**
     * Moves the content of source to path
     * This method works in mostly the manner
     * specified by the {@link java.nio.file.Files#move} method.
     * but:
     *  source and target are absolute paths of the same provider
     *  target does not exist
     *
     * @param source
     * @param target
     * @param options
     * @throws IOException
     */
    default void move( EightyPath source, EightyPath target, CopyOption... options ) throws IOException {
        Files.copy( source, target, options );
        Files.delete( source );
    }

    default boolean isSameFile( EightyPath path, Path path2 ) throws IOException {
        // works for all FS without links
        return false;
    }

    /**
     * Deletes the path as in {@link java.nio.file.Files#move}
     * but
     * - path is absolute
     * - is non empty
     * - exists
     * - is not root
     * @param path
     * @throws IOException
     */
    void delete( EightyPath path ) throws IOException;

    default Path toRealPath( Path norm ) {
        return norm;
    }

    default FileStore getFileStore( EightyPath path ) throws IOException {
        throw new UnsupportedOperationException();
    }

    default Iterable<FileStore> getFileStores() {
        throw new UnsupportedOperationException();
    }
    /**
     *
     * @param path
     * @param type A supported FileAttributeView
     * @param options
     * @param <V>
     * @return
     */
    <V extends FileAttributeView> V getFileAttributeView( EightyPath path, Class<V> type, LinkOption... options );

    default boolean isReadOnly() {
        return false;
    }

    default boolean isHidden(EightyPath paths) {
        return false;
    }

    default boolean isClosable() {
        return false;
    }

    default boolean watchable() {
        return false;
    }

    default boolean isReopenable() {
        return false;
    }

    default void reopen() {}

    default void setWatcher(EightyWatcher eightyFileSystem) {}

    default void createHardLink(EightyPath normLink, EightyPath normExisting) {
        throw new UnsupportedOperationException( "hard links not supported" );
    }
}
