package de.pfabulist.lindwurm.eighty.watch;

import de.pfabulist.lindwurm.eighty.EightyFS;
import de.pfabulist.lindwurm.eighty.path.EightyPath;

import java.io.IOException;
import java.nio.channels.Channel;
import java.nio.file.ClosedWatchServiceException;
import java.nio.file.Path;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.util.Collections;
import java.util.Set;
import java.util.concurrent.BlockingDeque;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.TimeUnit;

/**
 * ** BEGIN LICENSE BLOCK *****
 * BSD License (2 clause)
 * Copyright (c) 2006 - 2014, Stephan Pfab
 * All rights reserved.
 * <p>
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * <p>
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL Stephan Pfab BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * **** END LICENSE BLOCK ****
 */
public class EightyWatchService implements AttachedWatchService, Channel {

    private Set<EightyWatchKey> keyPool = Collections.newSetFromMap( new ConcurrentHashMap<>() );

    private final BlockingDeque<WatchKey> que = new LinkedBlockingDeque<>();

    private final EightyFS memoryFS;
    private Set<Thread> waiting = Collections.newSetFromMap( new ConcurrentHashMap<>() );

    private boolean open = true;

    public EightyWatchService( EightyFS memoryFS ) {
        this.memoryFS = memoryFS;
    }

    @Override
    public void close() throws IOException {
        open = false;
        for( Thread thread : waiting ) {
            thread.interrupt();
        }
        waiting.clear();

        for( EightyWatchKey key : keyPool ) {
            key.cancel();
        }

        keyPool.clear();
    }

    @Override
    public WatchKey poll() {
        return que.pollLast();
    }

    @Override
    public WatchKey poll( long timeout, TimeUnit unit ) throws InterruptedException {
        try {
            Thread current = Thread.currentThread();
            waiting.add( current );
            WatchKey ret = que.pollLast( timeout, unit );
            waiting.remove( current );
            return ret;
        } catch( InterruptedException ex ) {
            throw new ClosedWatchServiceException();
        }
    }

    @Override
    public WatchKey take() throws InterruptedException {
        try {
            Thread current = Thread.currentThread();
            waiting.add( current );
            WatchKey ret = que.takeLast();
            waiting.remove( current );
            return ret;
        } catch( InterruptedException ex ) {
            throw new ClosedWatchServiceException();
        }

    }

    @Override
    public EightyFS getFileSystem() {
        return memoryFS;
    }

    @Override
    public WatchKey register( EightyPath path, WatchEvent.Kind<?>... events ) {

        EightyWatchKey key = new EightyWatchKey( path, events );
        keyPool.add( key );
        return key;
    }

    @Override
    public boolean isOpen() {
        return open;
    }

    @Override
    public void signal( EightyPath path, WatchEvent.Kind<Path> event ) {
        for( EightyWatchKey key : keyPool ) {
            if( key.isWatchingFor( path.getParent(), event ) ) {
                synchronized( key ) {
                    if( key.isReady() ) {
                        que.addFirst( key );
                    }
                    key.signal( event, path );
                }
            }
        }
    }

}
