package de.pfabulist.lindwurm.eighty.path;

import de.pfabulist.kleinod.paths.Relativize;
import de.pfabulist.lindwurm.eighty.EightyFileSystem;
import de.pfabulist.lindwurm.eighty.PathConstraints;
import de.pfabulist.lindwurm.eighty.watch.AttachedWatchService;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.nio.file.*;
import java.util.*;
import java.util.stream.Collectors;

import static de.pfabulist.kleinod.emergent.Todo.todo;
import static de.pfabulist.lindwurm.eighty.path.ProviderPath.toRealPathEx;

/**
 * ** BEGIN LICENSE BLOCK *****
 * BSD License (2 clause)
 * Copyright (c) 2006 - 2014, Stephan Pfab
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL Stephan Pfab BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * **** END LICENSE BLOCK ****
 */

/**
 * EightyPath is a path of the form /a/b/c
 * the separator needs not be '/' but multiroot systems are supported, such as C:\, D:\
 */
public class EightyPath implements Path {
    private final EightyFileSystem fileSystem;
    private final List<String> elems;
    private final Optional<String> rootComponent;
    private final boolean absolute;

    private String string;
    private final boolean noDots;
    private boolean testedReal = false;

    public EightyPath( EightyFileSystem fileSystem, String first, String... more ) {
        this( fileSystem, new GetPathConverter( fileSystem.get80().getPathConstraints(), first, more ) );
    }

    protected EightyPath( EightyFileSystem eightyFileSystem, GetPathConverter gp ) {
        this( eightyFileSystem, gp.getRootComponent(), gp.isAbsolute(), gp.getAll() );
    }

    protected EightyPath( EightyFileSystem fileSystem, Optional<String> rootStr, boolean absolute, List<String> elems ) {
        this.fileSystem = fileSystem;
        this.elems = elems;
        this.rootComponent = rootStr;
        this.absolute = absolute;

        if( elems.size() > 1 && elems.contains( "" ) ) {
            throw new IllegalStateException( "empty path segment" );
        }

//        if ( elems.isEmpty() && !rootComponent.isPresent()) {
//            todo();
//        }
        if( toString().contains( "//" ) ) {
            todo();
        }

//        if ( absolute && elems.size() == 1 && elems.get(0).isEmpty()) {
//            todo();
//        }

        noDots = !elems.contains( "." ) && !elems.contains( ".." );
    }

    @Override
    public EightyFileSystem getFileSystem() {
        return fileSystem;
    }

    @Override
    public boolean isAbsolute() {
        return absolute;
    }

    @Override
    public Path getRoot() {
        if( absolute ) {
            return new EightyPath( fileSystem, rootComponent, true, Collections.emptyList() );
        }

        return null;
    }

    @Override
    public Path getFileName() {

        if( elems.isEmpty() && isAbsolute() ) {
            return null;
        }

        if( elems.size() <= 1 ) {
            if( isAbsolute() ) {
                return new EightyPath( fileSystem, Optional.empty(), false, elems );
            }
            return this;
        }

        return new EightyPath( fileSystem, Optional.empty(), false, elems.subList( elems.size() - 1, elems.size() ) );
    }

    public String getFileNameString() {

        if( elems.isEmpty() && isAbsolute() ) {
            return null;
        }

//        if( elems.size() <= 1 ) {
//            if( isAbsolute() ) {
//                return new EightyPath( fileSystem, Optional.empty(), false, elems );
//            }
//            return this;
//        }

        return elems.get( elems.size() - 1);
    }

    @Override
    public EightyPath getParent() {
        if( equals( getRoot() ) ) {
            return null;
        }

        if( !isAbsolute() && elems.size() <= 1 ) {
            return null;
        }

        return new EightyPath( fileSystem, rootComponent, absolute, elems.subList( 0, elems.size() - 1 ) );
    }

    @Override
    public int getNameCount() {
        return elems.size();
    }

    @Override
    public Path getName( int index ) {
        if( index < 0 || index >= elems.size() ) {
            throw new IllegalArgumentException( "index " + index + " <0 or >=  " + getNameCount() );
        }
        return new EightyPath( fileSystem, Optional.empty(), false, Collections.singletonList( elems.get( index ) ) );
    }

    @Override
    public Path subpath( int beginIndex, int endIndex ) {

        if( beginIndex < 0 || beginIndex >= elems.size() || beginIndex >= endIndex || endIndex > elems.size() ) {
            throw new IllegalArgumentException( "begin or end are not a legal interval [" + beginIndex + ", " + endIndex + ")" );
        }

        Path ret = new EightyPath( fileSystem, elems.get( beginIndex ) ); // todo rel ?

        for( int i = beginIndex + 1; i < endIndex; i++ ) {
            ret = ret.resolve( elems.get( i ) );
        }
        return ret;
    }

    @Override
    public boolean startsWith( Path other ) {
        if( isAbsolute() != other.isAbsolute() ) {
            return false;
        }

        for( int i = 0; i < other.getNameCount(); i++ ) {
            if( i >= getNameCount() ) {
                return false;
            }

            if( !getName( i ).equals( other.getName( i ) ) ) {
                return false;
            }
        }
        return true;
    }

    @Override
    public boolean startsWith( String other ) {
        return startsWith( getFileSystem().getPath( other ) );
    }

    @Override
    public boolean endsWith( Path other ) {
        for( int i = 0; i < other.getNameCount(); i++ ) {
            if( i >= getNameCount() ) {
                return false;
            }

            if( !getName( getNameCount() - 1 - i ).equals( other.getName( other.getNameCount() - 1 - i ) ) ) {
                return false;
            }
        }
        return true;
    }

    @Override
    public boolean endsWith( String other ) {
        return endsWith( getFileSystem().getPath( other ) );
    }

    @Override
    public EightyPath normalize() {
        List<String> newEls = new ArrayList<>();

        for( String elem : elems ) {
            switch( elem ) {
                case "..":
                    if( newEls.size() > 0 && !newEls.get( newEls.size() - 1 ).equals( ".." ) ) {
                        newEls.remove( newEls.size() - 1 );
                    } else {
                        if( !absolute ) {
                            newEls.add( ".." );
                        }
                    }
                    break;
                case ".":
                    // nix
                    break;
                case "":
                    // nix
                    break;
                default:
                    newEls.add( elem );
            }

        }

        if( !absolute && newEls.isEmpty() ) {
            return new EightyPath( fileSystem, rootComponent, false, Collections.singletonList( "" ) );
        }

        return new EightyPath( fileSystem, rootComponent, absolute, newEls );
    }

    @Override
    public Path resolve( Path other ) {

        if( !fileSystem.provider().equals( other.getFileSystem().provider() ) ) {
            throw new ProviderMismatchException();
        }

        if( other.isAbsolute() ) {
            return other;
        }

        if( !( other.getFileSystem() instanceof EightyFileSystem ) ) {
            throw new ProviderMismatchException( "Argument is from an other provider " + other );
        }

        EightyPath rother = (EightyPath) other;

        if( rother.rootComponent.isPresent() && !rootComponent.equals( rother.rootComponent ) ) {
            return other;
        }

        if( rother.elems.size() == 1 && rother.elems.get( 0 ).isEmpty() ) {
            return this;
        }

        if( elems.size() == 1 && elems.get( 0 ).isEmpty() ) {
            return other;
        }

        List<String> newEls = new ArrayList<>();
        newEls.addAll( elems );
        newEls.addAll( ( (EightyPath) other ).elems );

        Path ret = new EightyPath( fileSystem, rootComponent, absolute, newEls );

        return ret;
    }

    @Override
    public Path resolve( String other ) {
        return resolve( new EightyPath( fileSystem, other ) );
    }

    @Override
    public Path resolveSibling( Path other ) {
        if( other.isAbsolute() ) {
            return other;
        }

        Path parent = getParent();

        if( parent == null ) {
            return other;
        }

        return getParent().resolve( other );
    }

    @Override
    public Path resolveSibling( String other ) {
        return resolveSibling( new EightyPath( fileSystem, other ) );
    }

    @Override
    public Path relativize( Path other ) {
        if( isAbsolute() != other.isAbsolute() ) {
            throw new IllegalArgumentException( "'other' is different type of Path" );
        }

        return new EightyPath( fileSystem, Optional.empty(), false, Relativize.relativize( elems, ( (EightyPath) other ).elems ) );
    }

    @Override
    public URI toUri() {
        return URI.create( ( fileSystem.provider() ).getUriMapper().toURI(
                fileSystem.getId() )
                                   .toString() +
                                   toUriStyle() );
    }

    private String toUriStyle() {
        String ret = toAbsolutePath().toString();

        ret = ret.replace( '\\', '/' );
        ret = ret.replace( " ", "%20" );

        return ret;
    }

    @Override
    public EightyPath toAbsolutePath() {
        if( absolute ) {
            return this;
        }

        Optional<String> rc = Optional.ofNullable(
                rootComponent.orElseGet( () ->
                                                 fileSystem.get80().getPathConstraints().getRoots().stream().reduce( null, ( o, n ) -> n ) ) );

        return new EightyPath( fileSystem, rc, true, elems.size() == 1 && elems.get( 0 ).isEmpty() ? Collections.emptyList() : elems );
    }

    @Override
    public Path toRealPath( LinkOption... options ) throws IOException {
        Path real = toRealPathEx( this, options );

        if( !Files.exists( real ) ) {
            throw new NoSuchFileException( "File " + this + " does not exist" );
        }

        return real;

    }

    @Override
    public File toFile() {
        throw new UnsupportedOperationException( "only the default fs implements toFile" );
    }

    @Override
    public WatchKey register( WatchService watcher, WatchEvent.Kind<?>[] events, WatchEvent.Modifier... modifiers ) throws IOException {
        todo();
        return null;  //To change body of implemented methods use File | Settings | File Templates.
    }

    @Override
    public WatchKey register( WatchService watcher, WatchEvent.Kind<?>... events ) throws IOException {
        if( !( watcher instanceof AttachedWatchService ) ) {
            throw new ProviderMismatchException();
        }

        AttachedWatchService attachedWatcher = (AttachedWatchService) watcher;

        if( !fileSystem.get80().equals( attachedWatcher.getFileSystem() ) ) {
            throw new IllegalArgumentException();
        }

        if( !attachedWatcher.isOpen() ) {
            throw new ClosedWatchServiceException();
        }

        return attachedWatcher.register( this, events );
    }

    @Override
    public Iterator<Path> iterator() {
        return new Iterator<Path>() {
            private int idx = 0;

            @Override
            public boolean hasNext() {
                return idx < elems.size();
            }

            @Override
            public Path next() {
                Path ret = new EightyPath( fileSystem, elems.get( idx ) ); // getname ?
                idx++;
                return ret;
            }

            @Override
            public void remove() {
                todo();
            }
        };
    }

    @Override
    public int compareTo( Path other ) {
        if( !fileSystem.provider().equals( other.getFileSystem().provider() ) ) {
            throw new ClassCastException( "wrong provider, expected " + fileSystem.provider() + " got " + other.getFileSystem().provider() ); // todo root
        }
        // todo case insensitive
        return toString().compareTo( other.toString() );
    }

    @Override
    public String toString() {

        if ( string == null ) {

            string = ( rootComponent.orElse( "" ) ) +
                    ( absolute ? fileSystem.getSeparator() : "" ) +
                    elems.stream().collect( Collectors.joining( fileSystem.getSeparator()));
//                    elems.stream().reduce( ( a, b ) -> a + fileSystem.getSeparator() + b ).orElse( "" );
        }

        return string;
    }

    @Override
    public boolean equals( Object o ) {
        if( this == o ) {
            return true;
        }
        if( o == null || getClass() != o.getClass() ) {
            return false;
        }

        EightyPath other = (EightyPath) o;

        if( !rootComponent.equals( other.rootComponent ) ) {
            return false;
        }

        if( !fileSystem.equals( other.fileSystem ) ) {
            return false;
        }

        if( elems.size() != other.elems.size() ) {
            return false;
        }

        for( int i = 0; i < elems.size(); i++ ) {
            if( !fileSystem.get80().getPathConstraints().nameEquals( elems.get( i ), other.elems.get( i ) ) ) {
                return false;
            }
        }

        return true;
    }

    @Override
    public int hashCode() {
        PathConstraints ec = fileSystem.get80().getPathConstraints();
        int result = fileSystem.hashCode();
        result = elems.stream().map( ec::nameHash ).reduce( result, ( sofar, now ) -> 31 * sofar + now );
        result = 31 * result + rootComponent.hashCode();
        return result;
    }

    public Optional<String> getRootComponent() {
        return rootComponent;
    }

    public boolean knownReal() {
        return absolute && (testedReal || (noDots && !fileSystem.get80().supportsSymlinks()));
    }

    public void setTestedReal() {
        this.testedReal = true;
    }
}
