package de.pfabulist.lindwurm.eighty.attributes;

import de.pfabulist.lindwurm.eighty.EightyFS;
import de.pfabulist.lindwurm.eighty.EightyFileSystem;
import de.pfabulist.lindwurm.eighty.EightySymLink;
import de.pfabulist.lindwurm.eighty.RuntimeProxy;
import de.pfabulist.lindwurm.eighty.path.EightyPath;

import java.io.IOException;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributeView;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttributeView;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import static de.pfabulist.lindwurm.eighty.EightyUtils.existsEx;
import static de.pfabulist.lindwurm.eighty.path.ProviderPath.toRealPathEx;
import static de.pfabulist.unchecked.Unchecked.u;

/**
 * ** BEGIN LICENSE BLOCK *****
 * BSD License (2 clause)
 * Copyright (c) 2006 - 2015, Stephan Pfab
 * All rights reserved.
 * <p>
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * <p>
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL Stephan Pfab BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * **** END LICENSE BLOCK ****
 */

public class AttributeProvider {

    Map<String,RWAttributes> fromName = new HashMap<>();
    Map<Class<BasicFileAttributes>, RWAttributes> fromRead = new HashMap<>();
    Map<Class<FileAttributeView>, RWAttributes> fromView = new HashMap<>();


    public <V extends FileAttributeView> V getFileAttributeView( Path path, final Class<V> type, LinkOption... options ) {
        if ( !isViewSupported( type )) {
            return null;
        }

        if ( !existsEx( path, options ) ) {
            return getFileAttributeViewDummy( path, type );
        }

        EightyPath real = toRealPathEx( path, options );

        EightyFileSystem eightyFileSystem  = (EightyFileSystem) path.getFileSystem();
        EightyFS efs = eightyFileSystem.get80();


        V fav;

        Optional<EightySymLink> sym = efs.getSymlink( real );
        if( sym.isPresent() ) {
            fav = efs.getFileAttributeView( sym.get().getHost(), type );

            try {
                if( !( (BasicFileAttributeView) fav ).readAttributes().isSymbolicLink() ) {
                    if( !( fav instanceof LinkInfoSettable ) ) {
                        throw new UnsupportedOperationException( "the attribute view need to implement LinkInfoSettable in order to make SymLinks work" );
                    }

                    ( (LinkInfoSettable) ( fav ) ).setLink();
                }
            } catch( IOException e ) {
                throw u( e );
            }
        } else {
            fav = efs.getFileAttributeView( real, type );
        }

        if( eightyFileSystem.isReadOnly() ) {
            if( !( fav instanceof ReadonlySettable ) ) {
                throw new UnsupportedOperationException( "the attribute view need to implement ReadonlySettable in order to make Readonly Filesystems work" );
            }

            ( (ReadonlySettable) ( fav ) ).setReadonly();
        }

        return fav;
    }

    public <V extends FileAttributeView> boolean isViewSupported( Class<V> type ) {
        return fromView.containsKey( type );
    }

    public <R extends BasicFileAttributes> boolean isReadSupported( Class<R> type ) {
        return fromRead.containsKey( type );
    }

    public   <V extends FileAttributeView> V getFileAttributeViewDummy( Path path, final Class<V> type ) {
        return RuntimeProxy.of( type, name -> {
            if( name.equals( "name" ) ) {
                return "FileAttributeProxy";
            }

            throw new NoSuchFileException( path.toString() );
        } );

    }

    public <A extends BasicFileAttributes> A readAttributes( Path path, final Class<A> type, LinkOption... options ) throws IOException {
        if ( !isReadSupported( type )) {
            throw new UnsupportedOperationException( type + " not a supported FileAttributes class" );
        }

        if( !existsEx( path, options ) ) {
            throw new NoSuchFileException( path + " does not exist" );
        }

        // todo test custom atti, should fail if target fs does not have it


        return (A) getFileAttributeView( path,
                                         (Class<BasicFileAttributeView>)(Class)fromRead.get(type).getViewType(),
                                         options ).readAttributes();

    }

    public Map<String, Object> readAttributes( Path path, String attributes, LinkOption... options ) throws IOException {
        //EightyFS efs = checkProviderAndGet80( path );

        if( !existsEx( path, options ) ) {
            throw new NoSuchFileException( path + " does not exist" );
        }

        //EightyPath real = toRealPathEx( path, options );

        final String viewName = AttributeKeys.getName( attributes );

        RWAttributes rwAttributes = fromName.get( viewName );
        if ( rwAttributes == null ) {
            throw new UnsupportedOperationException( viewName + " not a supported FileAttributes class" );
        }

//        RWAttributes rwAttributes =
//                getAttributeNavigation( path ).getRWAttributesFromName( viewName ).
//                        orElseThrow( () -> new UnsupportedOperationException( viewName + " not a supported FileAttributes class" ) );


//        AttributeConnection<BasicFileAttributeView, BasicFileAttributes> connection =
//                real.getFileSystem().getAttributeProvider().
//                        getConnectionFromName( viewName ).orElseThrow( () -> new UnsupportedOperationException( viewName + " is not a supported FileAttributeView" ) );

        FileAttributeView view = getFileAttributeView( path,
                                                       rwAttributes.getViewType(),
                                                       options );

        Map<String, Object> ret = new HashMap<>();

        for( String key : AttributeKeys.getKeys( attributes, rwAttributes.getAttributeNames() ) ) {
            ret.put( key, rwAttributes.read( view, key ) );
        }

        return ret;
    }

    public void setAttribute( Path path, final String attribute, Object value, LinkOption... options ) throws IOException {

        // todo: link test // exiss test

        final String viewName = AttributeKeys.getName( attribute );

        RWAttributes rwAttributes = fromName.get( viewName );
        if ( rwAttributes == null ) {
            throw new UnsupportedOperationException( viewName + " not a supported FileAttributes class" );
        }

        Set<String> keys = AttributeKeys.getKeys( attribute, rwAttributes.getAttributeNames() );

        if( keys.size() != 1 ) {
            throw new IllegalArgumentException( "you can set only one attribute at a time" );
        }

        String attiName = keys.iterator().next();

        if ( !rwAttributes.isSettable( attiName )) {
            throw new IllegalArgumentException( "attribute not settable " + attiName );
        }

        FileAttributeView view = getFileAttributeView( path, rwAttributes.getViewType(), options );

        rwAttributes.set( view, attiName, value );
    }

    public Set<String> supportedFileAttributeViewNames() {
        return fromName.keySet();
    }
}
