package de.pfabulist.lindwurm.eighty.attributes;

import de.pfabulist.unchecked.functiontypes.BiConsumerE;
import de.pfabulist.unchecked.functiontypes.FunctionE;

import java.nio.file.attribute.BasicFileAttributeView;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.DosFileAttributeView;
import java.nio.file.attribute.DosFileAttributes;
import java.nio.file.attribute.FileAttributeView;
import java.nio.file.attribute.FileOwnerAttributeView;
import java.nio.file.attribute.FileTime;
import java.nio.file.attribute.GroupPrincipal;
import java.nio.file.attribute.PosixFileAttributeView;
import java.nio.file.attribute.PosixFileAttributes;
import java.nio.file.attribute.UserPrincipal;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Function;

import static de.pfabulist.unchecked.functiontypes.BiConsumerE.u;

/**
 * ** BEGIN LICENSE BLOCK *****
 * BSD License (2 clause)
 * Copyright (c) 2006 - 2015, Stephan Pfab
 * All rights reserved.
 * <p>
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * <p>
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL Stephan Pfab BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * **** END LICENSE BLOCK ****
 */

public class RWAttributesBuilder {

    public static final java.lang.String SIZE_NAME = "size";
    public static final java.lang.String CREATION_TIME_NAME = "creationTime";
    public static final java.lang.String LAST_ACCESS_TIME_NAME = "lastAccessTime";
    public static final java.lang.String LAST_MODIFIED_TIME_NAME = "lastModifiedTime";
    public static final java.lang.String FILE_KEY_NAME = "fileKey";
    public static final java.lang.String IS_DIRECTORY_NAME = "isDirectory";
    public static final java.lang.String IS_REGULAR_FILE_NAME = "isRegularFile";
    public static final java.lang.String IS_SYMBOLIC_LINK_NAME = "isSymbolicLink";
    public static final java.lang.String IS_OTHER_NAME = "isOther";

    private AttributeProvider provider = new AttributeProvider();

    public static RWAttributesBuilder attributes() {
        return new RWAttributesBuilder().addBasic();
    }

    public RWAttributesBuilder addBasic() {
        viewAndRead( "basic", BasicFileAttributeView.class, BasicFileAttributes.class ).
                attribute( LAST_MODIFIED_TIME_NAME,
                           BasicFileAttributes::lastModifiedTime,
                           u( ( ( view, val ) -> view.setTimes( (FileTime) val, null, null )))).
                attribute( LAST_ACCESS_TIME_NAME,
                           BasicFileAttributes::lastAccessTime,
                           u( ( view, val ) -> view.setTimes( null, (FileTime) val, null ) ) ).
                attribute(SIZE_NAME, BasicFileAttributes::size).
                attribute(CREATION_TIME_NAME, BasicFileAttributes::creationTime,
                          u((view, val) -> view.setTimes(null, null, (FileTime) val))).
                attribute(FILE_KEY_NAME, BasicFileAttributes::fileKey).
                attribute(IS_DIRECTORY_NAME, BasicFileAttributes::isDirectory).
                attribute(IS_REGULAR_FILE_NAME, BasicFileAttributes::isRegularFile).
                attribute(IS_OTHER_NAME, BasicFileAttributes::isOther).
                attribute( IS_SYMBOLIC_LINK_NAME, BasicFileAttributes::isSymbolicLink )
        ;

        return this;
    }

    public RWAttributesBuilder addPosix() {
        viewAndRead( "posix", PosixFileAttributeView.class, PosixFileAttributes.class ).
                attribute( "owner", PosixFileAttributes::owner, BiConsumerE.u(( v, g ) -> v.setOwner( (UserPrincipal) g ))).
                attribute( "group", PosixFileAttributes::group, BiConsumerE.u( ( v, g ) -> v.setGroup( (GroupPrincipal) g ))).
                attribute( "permissions", PosixFileAttributes::permissions, BiConsumerE.u( ( v, p ) -> v.setPermissions( (Set) p )));

        return this;
    }

    public RWAttributesBuilder addOwner() {
        viewOnly( "owner", FileOwnerAttributeView.class ).
                attribute( "owner", FunctionE.u( FileOwnerAttributeView::getOwner ) );

        return this;
    }

    public RWAttributesBuilder addDos() {
        viewAndRead( "dos", DosFileAttributeView.class, DosFileAttributes.class ).
                attribute( "hidden", DosFileAttributes::isHidden,
                           BiConsumerE.u( ( v, b ) -> v.setHidden( (Boolean) b ) ) ).
                attribute( "readonly", DosFileAttributes::isReadOnly ).
                attribute( "system", DosFileAttributes::isSystem ).
                attribute( "archive", DosFileAttributes::isArchive );

        return this;
    }

    public <V extends FileAttributeView> OneBuilder<V> viewOnly( String name, Class<V> viewClass ) {
        OneBuilder builder = new OneBuilder<>( name, viewClass, this );
        provider.fromName.put( name, builder.one );
        provider.fromView.put( (Class)viewClass, builder.one );
        return builder;
    }

    public <V extends BasicFileAttributeView, R extends BasicFileAttributes> TwoBuilder<V, R> viewAndRead( String name, Class<V> viewClass, Class<R> readerClass ) {
        TwoBuilder builder = new TwoBuilder<>( name, viewClass, readerClass, this );
        provider.fromName.put( name, builder.two );
        provider.fromRead.put( (Class<BasicFileAttributes>) readerClass, builder.two );
        provider.fromView.put( (Class)viewClass, builder.two );
        return builder;
    }

    public AttributeProvider build() {
        return provider;
    }

    public static class OneBuilder<V extends FileAttributeView> {

        private final OneClassAttributes one;
        private final RWAttributesBuilder builder;

        public OneBuilder( String name, Class<V> clazz, RWAttributesBuilder builder ) {
            this.builder = builder;
            one = new OneClassAttributes( name, clazz );
        }

        public OneBuilder<V> attribute( String name, Function<V, Object> get ) {
            one.reads.put( name, (Function<FileAttributeView, Object>) get );
            return this;
        }

        public OneBuilder<V> attribute( String name, Function<V, Object> get, BiConsumer<V, Object> set ) {
            one.reads.put( name, (Function<FileAttributeView, Object>) get );
            one.writes.put( name, (BiConsumer<FileAttributeView, Object>) set );
            return this;
        }

        public RWAttributesBuilder build() {
            return builder;
        }
    }

    public static class TwoBuilder<V extends BasicFileAttributeView, R extends BasicFileAttributes> {
        private final TwoClassAttributes two;
        private final RWAttributesBuilder builder;

        public TwoBuilder( String name, Class<V> viewClass, Class<R> readClass, RWAttributesBuilder builder ) {
            this.builder = builder;
            this.two = new TwoClassAttributes( name, viewClass, readClass );
        }

        public TwoBuilder<V,R> attribute( String name, Function<R, Object> get ) {
            two.reads.put( name, (Function<BasicFileAttributes, Object>) get );
            return this;
        }

        public TwoBuilder<V,R> attribute( String name, Function<R, Object> get, BiConsumer<V, Object> set ) {
            two.reads.put( name, (Function<BasicFileAttributes, Object>) get );
            two.writes.put( name, (BiConsumer<BasicFileAttributeView, Object>) set );
            return this;
        }

        public RWAttributesBuilder build() {
            return builder;
        }
    }


}
