package de.pfabulist.lindwurm.eighty;

import de.pfabulist.kleinod.paths.Pathss;
import de.pfabulist.lindwurm.eighty.attributes.RWAttributesBuilder;
import de.pfabulist.lindwurm.eighty.path.URIMapper;

import java.net.URI;
import java.nio.file.FileSystem;
import java.nio.file.FileSystemAlreadyExistsException;
import java.nio.file.FileSystemNotFoundException;
import java.nio.file.Path;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * ** BEGIN LICENSE BLOCK *****
 * BSD License (2 clause)
 * Copyright (c) 2006 - 2015, Stephan Pfab
 * All rights reserved.
 * <p>
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * <p>
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL Stephan Pfab BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * **** END LICENSE BLOCK ****
 */

public class ProviderURIStuff {

    private final URIMapper uriMapper;
    private final ConcurrentMap<String, EightyFileSystem> fileSystems = new ConcurrentHashMap<>();
    private final EightyFSCreator creator;

    public ProviderURIStuff( URIMapper uriMapper, EightyFSCreator creator ) {
        this.uriMapper = uriMapper;
        this.creator = creator;
    }

    public String getScheme() {
        return uriMapper.getScheme();
    }

    private void checkURI( URI uri ) {
        if( !uri.getScheme().equals( getScheme() ) ) {
            throw new IllegalArgumentException( "scheme does not fit filesystem, '" + getScheme() + "' expected got " + uri.getScheme() );
        }
    }

    public FileSystem getFileSystem( URI uri ) {
        checkURI( uri );

        String id = uriMapper.getSchemeSpecificPart( uri );

        FileSystem ret = fileSystems.get( id );

        if( ret == null ) {
            throw new FileSystemNotFoundException( uri.toString() );
        }

        if( !ret.isOpen() ) {
            fileSystems.remove( id ); // for GC
            throw new FileSystemNotFoundException( uri.toString() );
        }

        return ret;
    }

    public FileSystem newFileSystem( EightyProvider eightyProvider, URI uri, Map<String, ?> env2 ) {

        checkURI( uri );

        Map<String, Object> env = (Map) env2;

        String id = uriMapper.getSchemeSpecificPart( uri );

        if( id == null ) {
            throw new IllegalArgumentException( "scheme specific part is null : " + uri );
        }

        if( fileSystems.containsKey( id ) && fileSystems.get( id ).isOpen() ) {
            throw new FileSystemAlreadyExistsException( id );
        }

        Object fsid = uriMapper.fromString( id, env );

        RWAttributesBuilder attributesBuilder = RWAttributesBuilder.attributes();
        //AttributesBuilder attributesBuilder = attributes();
        EightyFS efs = creator.create( fsid, attributesBuilder, env );

        if( efs == null ) {
            // when the baseFileSystemCreator is stubbed only
            return null;
        }

        EightyFileSystem eightyFileSystem = new EightyFileSystem( efs, id, eightyProvider, attributesBuilder.build() );
        fileSystems.put( id, eightyFileSystem );

        efs.setWatcher( eightyFileSystem );

        return eightyFileSystem;
    }

    public Path getPath( URI uri ) {
        checkURI( uri );
        FileSystem fs = Pathss.getOrCreate( uri, Collections.<String, Object> emptyMap() );
        return fs.getPath( deUri( fs, uriMapper.getPathPart( uri ) ) );
    }

    private String deUri( FileSystem fs, String path ) {
        return path.replace( "/", fs.getSeparator() );
    }

    public URIMapper getUriMapper() {
        return uriMapper;
    }
}
