package de.pfabulist.lindwurm.eighty;

import de.pfabulist.kleinod.collection.Ref;
import de.pfabulist.lindwurm.eighty.path.EightyPath;
import de.pfabulist.unchecked.Unchecked;

import java.io.IOException;
import java.nio.file.*;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import static de.pfabulist.unchecked.Unchecked.u;
import static java.nio.file.LinkOption.NOFOLLOW_LINKS;

/**
 * ** BEGIN LICENSE BLOCK *****
 * BSD License (2 clause)
 * Copyright (c) 2006 - 2015, Stephan Pfab
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL Stephan Pfab BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * **** END LICENSE BLOCK ****
 */
public class EightyUtils {

    public static EightyFS get80( FileSystem fs ) {
        if ( fs instanceof EightyFileSystem ) {
            return ((EightyFileSystem) fs).get80();
        }

        throw new IllegalArgumentException("this is not an EightyFileSystem " + fs);
    }

    public static EightyFS get80( Path path ) {
        return get80( path.getFileSystem());
 }


    public static Stream<Path> getDirectoryStream(Path host)  {

        if (host.getFileSystem() instanceof EightyFileSystem) {
            return EightyUtils.get80(host.getFileSystem()).newDirectoryStream((EightyPath)host);
        }

//        try ( DirectoryStream<Path> dstr =  host.getFileSystem().provider().newDirectoryStream( host, k -> true )) {
//            return StreamSupport.stream( dstr.spliterator(), false );
//        }

        try (DirectoryStream<Path> dstr = Files.newDirectoryStream(host)) {
            List<Path> ret = new ArrayList<>();
            for ( Path kid : dstr ) {
                ret.add( kid );
            }

            return StreamSupport.stream(ret.spliterator(), false);
        } catch (IOException e) {
            throw u( e );
        }
    }

    public static Optional<String> checkFilenameLength(Path path, int maxFilenameLength) {
        for ( Path el : path ) {
            if ( el.toString().length() > maxFilenameLength ) {
                return Optional.of("path element too long, max length" + maxFilenameLength + " got " + el);
            }
        }

        return Optional.empty();

    }

    /**
     * same as Files.exists( NOFOLLOWLINKS )
     * but without checking attributes
     */
    public static boolean existsEx( Path path, LinkOption... options ) {
        if( Files.exists( path ) ) {
            return true;
        }

        if( options.length == 0 || !options[ 0 ].equals( NOFOLLOW_LINKS ) ) {
            return false;
        }

        try {
            Files.readSymbolicLink( path );
        } catch( IOException e ) {
            return false;
        }

        return true;
    }

//    void allowAccess( EightyPath path ) throws FileSystemException {
//        Optional<String> message = get80( path ).isPathAccessible( path );
//
//        if( message.isPresent() ) {
//            throw new FileSystemException( message.get() );
//        }
//    }

    static void isPathAccessible( Path path ) {
        Ref<String> reason = Ref.valueOf( "" );
        if ( !get80( path ).getPathConstraints().isPathAccessible( path, reason )) {
            throw Unchecked.u( new FileSystemException( reason.get() ) );
        }
    }



    public static String getFilenameString( Path path ) {
        if ( path instanceof EightyPath ) {
            return ((EightyPath)path).getFileNameString();
        }

        Path ret = path.getFileName();

        return ret == null ? null : ret.toString();
    }



}
