package de.pfabulist.lindwurm.eighty;

import de.pfabulist.lindwurm.eighty.attributes.AttributeProvider;
import de.pfabulist.unchecked.Filess;
import de.pfabulist.kleinod.paths.Pathss;
import de.pfabulist.lindwurm.eighty.path.EightyPath;
import de.pfabulist.lindwurm.eighty.path.URIMapper;

import java.io.IOException;
import java.net.URI;
import java.nio.channels.FileChannel;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.*;
import java.nio.file.attribute.BasicFileAttributeView;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileAttributeView;
import java.nio.file.attribute.FileTime;
import java.nio.file.spi.FileSystemProvider;
import java.time.Clock;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import static de.pfabulist.lindwurm.eighty.EightyUtils.existsEx;
import static de.pfabulist.lindwurm.eighty.EightyUtils.isPathAccessible;
import static de.pfabulist.lindwurm.eighty.ProviderChannel.newChannel;
import static de.pfabulist.lindwurm.eighty.path.ProviderPath.toRealPath;
import static de.pfabulist.lindwurm.eighty.path.ProviderPath.toRealPathEx;
import static java.nio.file.LinkOption.NOFOLLOW_LINKS;
import static java.nio.file.StandardCopyOption.COPY_ATTRIBUTES;
import static java.nio.file.StandardCopyOption.REPLACE_EXISTING;
import static java.util.Arrays.asList;

/**
 * ** BEGIN LICENSE BLOCK *****
 * BSD License (2 clause)
 * Copyright (c) 2006 - 2015, Stephan Pfab
 * All rights reserved.
 * <p>
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * <p>
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL Stephan Pfab BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * **** END LICENSE BLOCK ****
 */
public abstract class EightyProvider extends FileSystemProvider {


    private final ProviderURIStuff providerURIStuff;

    public EightyProvider( URIMapper uriMapper,
                           EightyFSCreator creator ) {
        providerURIStuff = new ProviderURIStuff( uriMapper, creator );
    }

    @Override
    public String getScheme() {
        return providerURIStuff.getScheme();
    }

    @Override
    public FileSystem newFileSystem( URI uri, Map<String, ?> env2 ) throws IOException {
        return providerURIStuff.newFileSystem( this, uri, env2 );
    }

    @Override
    public FileSystem getFileSystem( URI uri ) {
        return providerURIStuff.getFileSystem( uri );
    }

    @Override
    public Path getPath( URI uri ) {
        return providerURIStuff.getPath( uri );
    }


    @Override
    public SeekableByteChannel newByteChannel( Path pathArg, Set<? extends OpenOption> options, FileAttribute<?>... attrs ) throws IOException {
        final EightyFS eightyFS = checkProviderAndGet80( pathArg );
        return newChannel( pathArg, options, ( p, o ) -> eightyFS.newByteChannel( p, o, attrs ) );
    }

    @Override
    public FileChannel newFileChannel( Path pathArg, Set<? extends OpenOption> options, FileAttribute<?>... attrs ) throws IOException {
        final EightyFS eightyFS = checkProviderAndGet80( pathArg );
        return newChannel( pathArg, options, ( p, o ) -> eightyFS.newFileChannel( p, o, attrs ) );
    }



    @Override
    public DirectoryStream<Path> newDirectoryStream( Path dirArg, DirectoryStream.Filter<? super Path> filter ) throws IOException {
        checkProviderAndGet80( dirArg );
        EightyFileSystem fs = (EightyFileSystem) dirArg.getFileSystem();
        EightyPath dir = toRealPath( dirArg );

        if( !Files.exists( dir ) ) {
            throw new NoSuchFileException( "dir " + dir + " does not exist" );
        }

        // todo set time now or after close ?
        FileTime now = FileTime.from( Clock.systemUTC().instant() );

        if( !fs.isReadOnly() ) {
            Files.getFileAttributeView( dirArg, BasicFileAttributeView.class ).setTimes( null, now, null );
        }

        // todo: test for throw

        return fs.addClosable( new EightyDirectoryStream(
                dirArg,
                EightyUtils.getDirectoryStream( dir ),
                filter ) );
    }

    @Override
    public void createDirectory( Path dirArg, FileAttribute<?>... attrs ) throws IOException {
        EightyFS eighty = checkProviderAndGet80( dirArg );
        EightyPath dir = toRealPathEx( dirArg, NOFOLLOW_LINKS );

        if( existsEx( dir, NOFOLLOW_LINKS ) ) {
            throw new FileAlreadyExistsException( dir.toString() );
        }

        if( !Files.isDirectory( dir.getParent() ) ) {
            throw new NoSuchFileException( dir.getParent().toString() );
        }

        isPathAccessible( dir );

        eighty.createDirectory( dir, attrs );

        FileTime now = FileTime.from( Clock.systemUTC().instant() );
        Files.getFileAttributeView( dir.getParent(), BasicFileAttributeView.class ).setTimes( now, now, null );
    }

    @Override
    public void delete( Path pathArg ) throws IOException {
        EightyFS eightyFS = checkProviderAndGet80( pathArg );
        EightyPath path = toRealPathEx( pathArg, NOFOLLOW_LINKS );

        Optional<EightySymLink> sym = eightyFS.getSymlink( path );

        if( !Files.exists( path ) && !sym.isPresent() ) {
            throw new NoSuchFileException( "no such file " + pathArg );
        }

        if( path.getParent() == null ) {
            throw new IllegalArgumentException( "can't delete root" );
        }

        if( sym.isPresent() ) {
            eightyFS.delete( sym.get().getHost() );
            return;
        }

        if( Files.isDirectory( path ) && !Pathss.isEmpty( path ) ) {
            throw new DirectoryNotEmptyException( "dir not empty" + pathArg );
        }

        if( path.getFileSystem().isReadOnly() ) {
            throw new ReadOnlyFileSystemException();
        }

        // todo: write a test for throw

        eightyFS.delete( path );

        Files.setLastModifiedTime( path.getParent(), FileTime.from( Clock.systemUTC().instant() ) );
    }

    @Override
    public void copy( Path osource, Path otarget, CopyOption... options ) throws IOException {
        checkProviderAndGet80( otarget );
        EightyFS srcEightyFS = checkProviderAndGet80( osource );

        if( Files.isSameFile( osource, otarget ) ) {
            // no op
            return;
        }

        if( !asList( options ).contains( REPLACE_EXISTING ) && existsEx( otarget, NOFOLLOW_LINKS ) ) {
            throw new FileAlreadyExistsException( otarget.toString() );
        }

        if( Files.isDirectory( otarget ) && !Pathss.isEmpty( otarget ) ) {
            throw new DirectoryNotEmptyException( "" + otarget + " is not empty" );
        }

        if( osource.getFileSystem().isReadOnly() ) {
            throw new ReadOnlyFileSystemException();
        }

        if( existsEx( otarget, NOFOLLOW_LINKS ) ) {
            Files.delete( otarget );
        }

        EightyPath source = toRealPath( osource );
        EightyPath target = toRealPath( otarget );

        if( Files.isDirectory( source ) ) {
            Files.createDirectory( target );
            return;
        }

        srcEightyFS.copy( source, target, options );

        if( asList( options ).contains( COPY_ATTRIBUTES ) && !srcEightyFS.isReadOnly() ) { // todo link: where to set
            Files.setLastModifiedTime( target, Files.getLastModifiedTime( source ) );
        }
    }

    @Override
    public void move( Path osource, Path otarget, CopyOption... options ) throws IOException {
        EightyFS source80 = checkProviderAndGet80( osource );
        checkProviderAndGet80( otarget );

        EightyPath target = toRealPathEx( otarget, NOFOLLOW_LINKS );
        EightyPath source = toRealPathEx( osource, NOFOLLOW_LINKS );

        if( source.getParent() == null ) {
            throw new FileSystemException( "root can not be moved" );
        }

        if( !asList( options ).contains( REPLACE_EXISTING ) && Files.exists( target, NOFOLLOW_LINKS ) ) {
            throw new FileAlreadyExistsException( target.toString() );
        }

        if( Files.isDirectory( target ) && !Pathss.isEmpty( target ) ) {
            throw new DirectoryNotEmptyException( target.toString() );
        }

        if( !Files.exists( target.getParent() ) ) {
            throw new NoSuchFileException( "parent of target is missing " + otarget );
        }

        isPathAccessible( target );

        if( osource.getFileSystem().isReadOnly() ) {
            throw new ReadOnlyFileSystemException();
        }

        Files.deleteIfExists( target ); // todo if target is sym link ?

        // todo target link
        // todo: ! hard link dirs can make this interesting

        Path parent = target.getParent();

        while( parent != null ) {
            if( parent.equals( source ) ) {
                throw new FileSystemException( "a directory can not be moved into one of its own subdirectories" );
            }
            parent = parent.getParent();
        }

        if( Files.isSymbolicLink( source ) ) {
            Files.createSymbolicLink( target, source80.getSymlink( source ).get().getTarget() );
            Files.delete( source );
            return;
        }

        source80.move( source, target );
        Filess.setLastModifiedTime( source.getParent(), FileTime.from( Clock.systemUTC().instant() ) );
    }

    @Override
    public boolean isSameFile( Path opath, Path opath2 ) throws IOException {

        if( !opath.getFileSystem().provider().equals( opath2.getFileSystem().provider() ) ) {
            return false;
        }

        checkProviderAndGet80( opath );

        if( opath.equals( opath2 ) ) {
            return true;
        }

        if( !Files.exists( opath ) &&  Files.exists( opath2 ) ) {
            return false;
        }

        if( !Files.exists( opath2 ) ) {
            if( Files.exists( opath ) ) {
                return false;
            }

            Path r1 = toRealPathEx( opath, NOFOLLOW_LINKS );
            Path r2 = toRealPathEx( opath2, NOFOLLOW_LINKS );

            return r1.equals( r2 );
        }

        // eightyfilesystems supporting hardlinks must support filekey
        Object fk1 = Files.readAttributes( opath, BasicFileAttributes.class ).fileKey();
        Object fk2 = Files.readAttributes( opath2, BasicFileAttributes.class ).fileKey();

        if( fk1 != null && fk2 != null && fk1.equals( fk2 ) ) {
            return true;
        }

        Path r1 = opath.toRealPath();
        Path r2 = opath2.toRealPath();

        // if there no hardlinks to dirs, path to file is now unique
        return r1.equals( r2 );

////        if ( isOtherFileSystem( r1, path)) {
////            return Files.isSameFile( r1, r2 );
////        }
//
//        if ( !r1.equals( opath ) || !r2.equals( opath2 )) {
//            return isSameFile( r1, r2 );
//        }
//
//        if ( r1.equals( r2 )) {
//            return true;
//        }
//
//        return false;
    }

    @Override
    public boolean isHidden( Path path ) throws IOException {
        checkProviderAndGet80( path );

//        if ( !Files.exists( path )) {
//            throw new NoSuchFileException( path.toString() );
//        }

        //todo sym links ?

        return checkProviderAndGet80( path ).isHidden( toRealPath( path ) );
    }

    @Override
    public FileStore getFileStore( Path path ) throws IOException {
        checkProviderAndGet80( path );

        if( !Files.exists( path ) ) {
            throw new NoSuchFileException( path.toString() );
        }

        return checkProviderAndGet80( path ).getFileStore( toRealPath( path ) );
    }

    @Override
    public void checkAccess( Path pathArg, AccessMode... modes ) throws IOException {

        EightyFS efs = checkProviderAndGet80( pathArg );
        EightyPath path = toRealPath( pathArg );

        efs.checkAccess( path, modes );
    }

    @Override
    public <V extends FileAttributeView> V getFileAttributeView( Path path, final Class<V> type, LinkOption... options ) {
        checkProviderAndGet80( path );
        EightyFileSystem  eightyFileSystem  = (EightyFileSystem) path.getFileSystem();
        AttributeProvider attributeProvider = eightyFileSystem.getAttributeProvider();


        return attributeProvider.getFileAttributeView( path, type, options );
    }


    @Override
    public <A extends BasicFileAttributes> A readAttributes( Path path, final Class<A> type, LinkOption... options ) throws IOException {
        checkProviderAndGet80( path );
        EightyFileSystem  eightyFileSystem  = (EightyFileSystem) path.getFileSystem();

        AttributeProvider attributeProvider = eightyFileSystem.getAttributeProvider();

        return attributeProvider.readAttributes( path, type, options );
    }

    @Override
    public Map<String, Object> readAttributes( Path path, String attributes, LinkOption... options ) throws IOException {
        checkProviderAndGet80( path );
        EightyFileSystem eightyFileSystem = (EightyFileSystem) path.getFileSystem();
        AttributeProvider attributeProvider = eightyFileSystem.getAttributeProvider();

        return attributeProvider.readAttributes( path, attributes, options );
    }

    @Override
    public void setAttribute( Path path, final String attribute, Object value, LinkOption... options ) throws IOException {
        checkProviderAndGet80( path );
        EightyFileSystem eightyFileSystem = (EightyFileSystem) path.getFileSystem();
        AttributeProvider attributeProvider = eightyFileSystem.getAttributeProvider();

        attributeProvider.setAttribute( path, attribute, value, options );
    }

    @Override
    public void createLink( Path link, Path existing ) throws IOException {

        EightyFS efs = checkProviderAndGet80( existing );
        checkProviderAndGet80( link );

        if( existsEx( link, NOFOLLOW_LINKS ) ) {
            throw new FileAlreadyExistsException( link.toString() );
        }

        if( !link.getFileSystem().equals( existing.getFileSystem() ) ) {
            throw new FileSystemException( "cannot link to a different filesystem" );
        }

        if( link.getFileSystem().isReadOnly() ) {
            throw new ReadOnlyFileSystemException();
        }

        EightyPath rLink = toRealPath( link );

        isPathAccessible( rLink );

        EightyPath rExisting = toRealPath( existing );

        efs.createHardLink( rLink, rExisting );

    }

    @Override
    public void createSymbolicLink( Path link, Path target, FileAttribute<?>... attrs ) throws IOException {

        EightyFS efs = checkProviderAndGet80( link );
        checkProviderAndGet80( target );

        EightyPath normLink = toRealPath( link );

        if( Files.exists( normLink ) ) {
            throw new FileAlreadyExistsException( link.toString() );
        }

        isPathAccessible( normLink );

        if( link.getFileSystem().isReadOnly() ) {
            throw new ReadOnlyFileSystemException();
        }

        efs.createSymLink( normLink, (EightyPath) target, attrs );
    }

    @Override
    public Path readSymbolicLink( Path link ) throws IOException {
        EightyFS efs = checkProviderAndGet80( link );
        EightyPath real = toRealPathEx( link, NOFOLLOW_LINKS );

        // todo: no such file

        Optional<EightySymLink> sym = efs.getSymlink( real );

        if( sym.isPresent() ) {
            return sym.get().getTarget();
        }

        throw new NotLinkException( link.toString() );
    }

    public URIMapper getUriMapper() {
        return providerURIStuff.getUriMapper();
    }

    /*
     * -------------------------------------- private --------------------------------------------------------------
     */

    private boolean isOtherFileSystem( Path nn, EightyPath old ) {
        return !nn.getFileSystem().equals( old.getFileSystem() );
    }


    private EightyFS checkProviderAndGet80( Path path ) {
        if( path.getFileSystem().provider() != this ) {
            throw new ProviderMismatchException( "expected path for " + getScheme() + " got " + path.getFileSystem().provider().getScheme() );
        }

        if( !path.getFileSystem().isOpen() ) {
            throw new ClosedFileSystemException();
        }

        return ( (EightyFileSystem) path.getFileSystem() ).get80();
    }


}
