package de.pfabulist.lindwurm.eighty;

import com.esotericsoftware.minlog.Log;
import de.pfabulist.lindwurm.eighty.attributes.AttributeProvider;
import de.pfabulist.lindwurm.eighty.close.CloseWithMe;
import de.pfabulist.lindwurm.eighty.path.EightRegexMatcher;
import de.pfabulist.lindwurm.eighty.path.EightyPath;
import de.pfabulist.lindwurm.eighty.path.GlobToRegex;
import de.pfabulist.lindwurm.eighty.watch.EightyWatchService;
import de.pfabulist.lindwurm.eighty.watch.EightyWatcher;

import java.io.IOException;
import java.nio.channels.Channel;
import java.nio.file.*;
import java.nio.file.attribute.UserPrincipalLookupService;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import static de.pfabulist.kleinod.emergent.Todo.todo;

/**
 * ** BEGIN LICENSE BLOCK *****
 * BSD License (2 clause)
 * Copyright (c) 2006 - 2015, Stephan Pfab
 * All rights reserved.
 * <p>
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * <p>
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL Stephan Pfab BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * **** END LICENSE BLOCK ****
 */
public class EightyFileSystem extends FileSystem implements EightyWatcher {

    private final EightyFS eightyFS;
    private final EightyProvider provider;
    private final CloseWithMe closeWithMe = new CloseWithMe();
    private final String id;
    private boolean open = true;
    private List<EightyWatchService> watchServices = new ArrayList<>();
    private AttributeProvider attributeProvider;

    public EightyFileSystem( EightyFS eightyFileSystem,
                             String id,
                             EightyProvider provider,
                             AttributeProvider attributeProvider ) {
        this.eightyFS = eightyFileSystem;
        this.provider = provider;
        this.id = id;
        this.attributeProvider = attributeProvider;

        Runtime.getRuntime().addShutdownHook( new Thread( () -> {
            try {
                if( eightyFS.isClosable() ) {
                    close();
                }
            } catch( Throwable e ) {
                Log.error( "exception in shutdown hook: " + e );
            }
        } ) );

    }

    @Override
    public EightyProvider provider() {
        return provider;
    }

    @Override
    public void close() throws IOException {
        if( !open ) {
            return;
        }

        if( !eightyFS.isClosable() ) {
            throw new UnsupportedOperationException( "FileSystem " + this + " can not be closed" );
        }

        eightyFS.onClose();

        open = false;
        closeWithMe.close();

        Log.info( "closed " + this );

    }

    @Override
    public boolean isOpen() {
        return open;

    }

    @Override
    public boolean isReadOnly() {
        return eightyFS.isReadOnly();
    }

    @Override
    public String getSeparator() {
        return eightyFS.getPathConstraints().getMainSeparator();
    }

    @Override
    public Iterable<Path> getRootDirectories() {

        List<String> roots = eightyFS.getPathConstraints().getRoots();

        if( roots.isEmpty() ) {
            Path root = new EightyPath( this, getSeparator() );
            return Collections.singletonList( root );
        }

        List<Path> ret = new ArrayList<>();

        for( String root : roots ) {
            ret.add( new EightyPath( this, root + getSeparator() ) );
        }

        return ret;
    }

    @Override
    public Iterable<FileStore> getFileStores() {
        return eightyFS.getFileStores();
    }

    @Override
    public Set<String> supportedFileAttributeViews() {
        return attributeProvider.supportedFileAttributeViewNames();
    }

    @Override
    public Path getPath( String first, String... more ) {
        return new EightyPath( this, first, more );
    }

    private final static String REGEX = "regex:";
    private static final String GLOB = "glob:";

    @Override
    public PathMatcher getPathMatcher( String syntaxAndPattern ) {
        if( syntaxAndPattern.toLowerCase().startsWith( REGEX ) ) {
            return new EightRegexMatcher( syntaxAndPattern.substring( REGEX.length() ) );
        }

        // TODO
        if( syntaxAndPattern.toLowerCase().startsWith( GLOB ) ) {
            return new EightRegexMatcher( GlobToRegex.toRegexPattern( syntaxAndPattern.substring( GLOB.length() ) ) );
        }

        todo();
        return null;  //To change body of implemented methods use File | Settings | File Templates.
    }

    @Override
    public UserPrincipalLookupService getUserPrincipalLookupService() {
        return eightyFS.getUserPrincipalLookupService();
    }

    @Override
    public WatchService newWatchService() throws IOException {
        if( !eightyFS.watchable() ) {
            throw new UnsupportedOperationException();
        }

        EightyWatchService ws = new EightyWatchService( eightyFS );
        watchServices.add( ws );
        return addClosable( ws );
    }

    public EightyFS get80() {
        return eightyFS;
    }


    public <T extends Channel> T addClosable( T cl ) {
        closeWithMe.add( cl );
        return cl;
    }

    @Override
    public void signal( EightyPath path, WatchEvent.Kind<Path> eventKind ) {
        for( EightyWatchService ws : watchServices ) {
            ws.signal( path, eventKind );
        }
    }

    public void reopen() {

        if( open ) {
            return;
        }

        if( !eightyFS.isReopenable() ) {
            throw new IllegalArgumentException( "eightfs is not reopenable" );
        }

        open = true;
        eightyFS.reopen();

        Log.info( "reopened " + this );
    }

    @Override
    public String toString() {
        return "EightFileSystem with " + eightyFS;
    }

    public String getId() {
        return id;
    }

    public AttributeProvider getAttributeProvider() {
        return attributeProvider;
    }
}


