package de.pfabulist.lindwurm.eighty;

import de.pfabulist.unchecked.Filess;
import de.pfabulist.kleinod.paths.Pathss;
import de.pfabulist.lindwurm.eighty.path.EightyPath;
import de.pfabulist.lindwurm.eighty.watch.EightyWatcher;

import java.io.IOException;
import java.io.InputStream;
import java.nio.channels.FileChannel;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.AccessMode;
import java.nio.file.CopyOption;
import java.nio.file.FileStore;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileAttributeView;
import java.nio.file.attribute.FileTime;
import java.nio.file.attribute.UserPrincipalLookupService;
import java.util.*;
import java.util.stream.Stream;

import static de.pfabulist.unchecked.Unchecked.u;

/**
 * ** BEGIN LICENSE BLOCK *****
 * BSD License (2 clause)
 * Copyright (c) 2006 - 2015, Stephan Pfab
 * All rights reserved.
 * <p>
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * <p>
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL Stephan Pfab BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * **** END LICENSE BLOCK ****
 */

/**
 * EightFs is an interface that helps implement NIO2 filesystems
 * <p>
 * the methods can assume that all Path arguments are from this filesystem and absolute
 */
public interface EightyFS {

    /**
     * Creates new SeekableByteChannel to that path
     * This method works in mostly the manner
     * specified by the {@link java.nio.file.Files#newByteChannel} method.
     * <p>
     * Differences to the general method:
     * - The set of options is not empty
     * - APPEND and READ are not both in the options
     * - IF the file does not exist than WRITE and either CREATE or CREATE_NEW are in the options
     * - If READ and no Create variants are in the options than the file exists
     * - The path is not a directory
     * - The path is absolute, normalized and has no sym link in its path
     * - The channel needs to implement InformativeClosable
     * - Parent path exists and is a directory
     *
     * @param path    An absolute path of a filesystem of this provider
     * @param options A non empty set of options without any conflicts, see above
     * @param attrs   attributes
     * @return
     */
    FileChannel newFileChannel( EightyPath path, Set<? extends OpenOption> options, FileAttribute<?>... attrs );

    default SeekableByteChannel newByteChannel( EightyPath path, Set<? extends OpenOption> options, FileAttribute<?>... attrs ) {
        return newFileChannel( path, options, attrs );
    }



    /**
     * Create a new Directory
     * This method should work in the manner specified in the {@link java.nio.file.Files#createDirectory} method.
     * <p>
     * Differences are
     * - dir is absolute
     * - dirs parent is a dir and exists
     * - dir does not exist
     *
     * @param dir
     * @param attrs
     */
    void createDirectory( EightyPath dir, FileAttribute<?>... attrs );

    /**
     * Check Read, Write, Exec access
     * is called quite often for file existence
     */
    void checkAccess( EightyPath path, AccessMode... modes );

    Stream<Path> newDirectoryStream( EightyPath dir );

    /**
     * Copy with same provider (not necessarily same FS)
     * target does not exist
     * needs to send out watch events (if supported in FS)
     *
     * @param source
     * @param target
     * @param options
     */
    default void copy( EightyPath source, EightyPath target, CopyOption... options ) {
        try {
            try( InputStream in = Files.newInputStream( source ) ) {
                Files.copy( in, target );
            }
        } catch( IOException e ) {
            throw u( e );
        }
    }

    /**
     * Moves the content of source to path
     * This method works in mostly the manner
     * specified by the {@link java.nio.file.Files#move} method.
     * but:
     * source and target are absolute paths of the same filesystem
     * source path is sym link free
     * target does not exist
     * target parent does exist
     *
     * @param source
     * @param target
     */
    default void move( EightyPath source, EightyPath target ) {
        FileTime lastModi = Filess.getLastModifiedTime( source );
        Pathss.copyRecursive( source, target );
        Pathss.deleteRecursive( source );
        Filess.setLastModifiedTime( target, lastModi );
    }

    /**
     * Deletes the path as in {@link java.nio.file.Files#move}
     * but
     * - path is absolute
     * - is non empty
     * - exists
     * - is not root
     *
     * @param path
     */
    void delete( EightyPath path );

    default FileStore getFileStore( EightyPath path ) {
        throw new UnsupportedOperationException();
    }

    default Iterable<FileStore> getFileStores() {
        throw new UnsupportedOperationException();
    }

    /**
     * @param path
     * @param type A supported FileAttributeView
     * @param <V>
     * @return
     */
    <V extends FileAttributeView> V getFileAttributeView( EightyPath path, Class<V> type );

    default boolean isReadOnly() {
        return false;
    }

    default void setReadOnly( boolean readOnly ) {}

    default boolean isHidden( EightyPath paths ) {
        return false;
    }

    default boolean isClosable() {
        return false;
    }

    default boolean watchable() {
        return false;
    }

    default boolean isReopenable() {
        return false;
    }

    default void reopen() {}

    default void setWatcher( EightyWatcher eightyFileSystem ) {
    }

    default void createHardLink( EightyPath normLink, EightyPath normExisting ) {
        throw new UnsupportedOperationException( "hard links not supported" );
    }

    default void createSymLink( EightyPath link, EightyPath existing, FileAttribute<?>[] attrs ) {
        throw new UnsupportedOperationException();
    }

    default Optional<EightySymLink> getSymlink( EightyPath path ) {
        return Optional.empty();
    }

    default boolean supportsSymlinks() {
        return false;
    }

    default void onClose() {
    }

    default UserPrincipalLookupService getUserPrincipalLookupService() {
        throw new UnsupportedOperationException( "not implemented" );
    }

    default PathConstraints getPathConstraints() {
        return PathConstraints.UnixUnlimited;
    }

    default void newContent( EightyPath path ) {}
}
