package de.pco.imageio;

/*-
 * #%L
 * pco-imageio
 * %%
 * Copyright (C) 2020 PCO
 * %%
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 * #L%
 */

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertThrows;

import java.awt.image.BufferedImage;
import java.awt.image.DataBuffer;
import java.awt.image.DataBufferByte;
import java.awt.image.PixelInterleavedSampleModel;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.List;

import javax.imageio.ImageIO;

import org.junit.jupiter.api.Test;

import de.pco.common.AbstractImageData;

/**
 * Tests the RawImageReader for reading the raw data. 
 * @author PCO
 *
 */
public class RawImageReaderTest 
{
    /**
     * Tests the reading from the RawImageInputStream over an external byte array - simulates 
     * the UByteArray returned from Camera.getImageCompressed(). Saves the result to
     * C:\Users\{user}\AppData\Local\Temp\pco-testReadCompressed-{...}.png
     * for the manual check.
     * @throws IOException
     */
	@Test
	void testReadCompressed() throws IOException 
	{
	    URL resource = getClass().getResource("/test.bmp");
		BufferedImage bi = ImageIO.read(resource);
	 	
		byte[] data = ((DataBufferByte)bi.getRaster().getDataBuffer()).getData();
		ByteArrayInputStream bais = new ByteArrayInputStream(data);
		RawImageInputStream riis = new RawImageInputStream(ImageIO.createImageInputStream(bais), 
			bi.getWidth(), bi.getHeight(), true);
		RawImageReader reader = new RawImageReader();
		reader.setInput(riis);
		
		BufferedImage bi2 = reader.read(0);
		assertNotNull(bi2);
		assertEquals(DataBuffer.TYPE_BYTE, bi2.getSampleModel().getDataType());
		assertEquals(PixelInterleavedSampleModel.class, bi2.getSampleModel().getClass());
		
		File file = File.createTempFile("pco-testReadCompressed-", ".png");
		ImageIO.write(bi, "png", file);
		//visually check image C:\Users\{user}\AppData\Local\Temp\pco-testReadCompressed-{...}.png
	}
	
	/**
	 * Tests whether the stream is properly reseted when the user tries to read 2x the same image
	 * @throws IOException
	 */
	@Test
    void testReadSecondPass() throws IOException 
    {
        URL resource = getClass().getResource("/test.bmp");
        BufferedImage bi = ImageIO.read(resource);
        
        byte[] data = ((DataBufferByte)bi.getRaster().getDataBuffer()).getData();
        ByteArrayInputStream bais = new ByteArrayInputStream(data);
        RawImageInputStream riis = new RawImageInputStream(ImageIO.createImageInputStream(bais), 
            bi.getWidth(), bi.getHeight(), true);
        RawImageReader reader = new RawImageReader();
        reader.setInput(riis);
        
        BufferedImage bi2 = reader.read(0);
        BufferedImage bi3 = reader.read(0);
        assertNotNull(bi3);
        assertEquals(DataBuffer.TYPE_BYTE, bi3.getSampleModel().getDataType());
        assertEquals(PixelInterleavedSampleModel.class, bi3.getSampleModel().getClass());
    }
	
	/**
	 * Tests the reader whether it reads the image from the multi-image RawImageInputStream.
	 * Check the png file
	 * C:\Users\{user}\AppData\Local\Temp\pco-testReaderOnMultiStream-{...}.png
	 * @throws IOException
	 */
	@Test
	void testReaderOnMultiStream() throws IOException
	{
	    List<AbstractImageData> aidList = TestUtils.getTestImages();
        RawImageInputStream riis = new RawImageInputStream(aidList);
        RawImageReader reader = new RawImageReader();
        reader.setInput(riis);
        BufferedImage bi1 = reader.read(0);
        BufferedImage bi2 = reader.read(1);
        BufferedImage bi3 = reader.read(2);
        
        File file = File.createTempFile("pco-testReaderOnMultiStream-", ".png");
        ImageIO.write(bi1, "png", file);
        file = File.createTempFile("pco-testReaderOnMultiStream-", ".png");
        ImageIO.write(bi2, "png", file);
        file = File.createTempFile("pco-testReaderOnMultiStream-", ".png");
        ImageIO.write(bi3, "png", file);
        
        // change input
        URL resource = getClass().getResource("/test.bmp");
        BufferedImage bi = ImageIO.read(resource);
        byte[] data = ((DataBufferByte)bi.getRaster().getDataBuffer()).getData();
        ByteArrayInputStream bais = new ByteArrayInputStream(data);
        RawImageInputStream riis2 = new RawImageInputStream(ImageIO.createImageInputStream(bais), 
            bi.getWidth(), bi.getHeight(), true);
        reader.setInput(riis2);
        assertThrows(IndexOutOfBoundsException.class, () -> reader.read(1));
        BufferedImage bi4 = reader.read(0);
        file = File.createTempFile("pco-testReaderOnMultiStream-", ".png");
        ImageIO.write(bi4, "png", file);
	}
}
