package de.pco.imageio;

import static org.junit.jupiter.api.Assertions.assertDoesNotThrow;
import static org.junit.jupiter.api.Assertions.assertEquals;

/*-
 * #%L
 * pco-imageio
 * %%
 * Copyright (C) 2020 PCO
 * %%
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 * #L%
 */

import static org.junit.jupiter.api.Assertions.assertNull;
import static org.junit.jupiter.api.Assertions.assertThrows;

import java.awt.image.BufferedImage;
import java.awt.image.DataBufferByte;
import java.io.IOException;
import java.net.URL;
import java.util.List;

import javax.imageio.ImageIO;

import org.junit.jupiter.api.Test;

import de.pco.common.AbstractImageData;
import de.pco.common.ImageDataCompressed;
import de.pco.common.UByteArray;

/**
 * Tests the RawImageInputStream
 * @author PCO
 */
class RawImageInputStreamTest 
{
    /**
     * Tests the {@link RawImageInputStream} over a single image - constructor, openSubstream 
     * (which is not necessary by single image), closing.
     * 
     * @throws IOException
     */
    @Test
    void testSingeImageInStream() throws IOException
    {
        URL resource = getClass().getResource("/test.bmp");
        BufferedImage bi = ImageIO.read(resource);
        byte[] data = ((DataBufferByte)bi.getRaster().getDataBuffer()).getData();
        
        UByteArray ubb = new UByteArray(data);
        
        ImageDataCompressed idc = new ImageDataCompressed(ubb, bi.getWidth(), bi.getHeight(), null);
        
        RawImageInputStream riis = new RawImageInputStream(idc);
        assertDoesNotThrow(() -> riis.openSubstreamOnImage(0)); // unnecessary, but it should not mind
        assertDoesNotThrow(() -> riis.readInt());
        riis.close();
        assertThrows(IllegalStateException.class, () -> riis.readInt());
    }

    /**
     * Tests the RawImageInputStream over multiple images, checks for proper
     * substream opening. 
     * @throws IOException
     */
    @Test
    void testMultiImageInStream() throws IOException
    {
        List<AbstractImageData> aidList = TestUtils.getTestImages();
        
        RawImageInputStream riis = new RawImageInputStream(aidList);
        assertEquals(4, riis.getNumImages());
        assertNull(riis.getMetadata(3));
        assertThrows(IndexOutOfBoundsException.class, 
            () -> riis.getMetadata(4));
        assertThrows(IllegalStateException.class, 
            () -> riis.readInt());
        riis.openSubstreamOnImage(3);
        assertDoesNotThrow(() -> riis.readInt()); 
    }
}
