package de.pco.imageio;

/*-
 * #%L
 * pco-imageio
 * %%
 * Copyright (C) 2020 PCO
 * %%
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 * #L%
 */

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNotNull;

import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.net.URL;

import javax.imageio.IIOImage;
import javax.imageio.ImageIO;
import javax.imageio.ImageTypeSpecifier;
import javax.imageio.metadata.IIOMetadata;
import javax.imageio.stream.ImageInputStream;

import org.junit.jupiter.api.Test;

import com.twelvemonkeys.imageio.plugins.tiff.TIFFImageMetadata;
import com.twelvemonkeys.imageio.plugins.tiff.TIFFImageReader;
import com.twelvemonkeys.imageio.plugins.tiff.TIFFImageReaderSpi;

import de.pco.sdk.BildStruct;
import de.pco.sdk.enums.CameraSyncMode;
import de.pco.sdk.enums.ImageType;
import de.pco.sdk.enums.SyncStatus;
import de.pco.sdk.enums.TriggerMode;

/**
 * Tests the B16ImageWriter class
 * @author PCO
 *
 */
public class B16ImageWriterTest 
{
    /**
     * Test whether the metadata of an image is written correctly by means of 
     * loading the written image once again. It is necessary to have working B16ImageReader 
     * @throws IOException
     */
    @Test
    void testWrite() throws IOException 
    {
        
        URL resource = getClass().getResource("/test.b16");
        B16ImageReader reader = new B16ImageReader();
        ImageInputStream iis = ImageIO.createImageInputStream(resource.openConnection().getInputStream()); 
        reader.setInput(iis);
        IIOImage image = reader.readAll(0, null);
        PcoIIOMetadata originalMetadata = (PcoIIOMetadata)image.getMetadata();
        BildStruct originalMbean = originalMetadata.getBildStruct();
        assertNotNull(image);
     
        B16ImageWriter writer = new B16ImageWriter();
        File file = File.createTempFile("pco-testWrite-", ".b16");
        writer.setOutput(ImageIO.createImageOutputStream(file));
        writer.write(image);
        
        reader.setInput(ImageIO.createImageInputStream(file));
        image = reader.readAll(0, null);
        PcoIIOMetadata metadata = (PcoIIOMetadata)image.getMetadata();
        BildStruct mBean = metadata.getBildStruct();
        
        assertEquals(mBean, originalMbean);
    }
    
    /**
     * Tests the convertImageMetadata method. Loads a TIFF image
     * written by PCO Camware software that included the proprietary metadata
     * stated in the MetadataBean class of PCO.java
     */
    @Test
    void testConvertImageMetadata() throws IOException 
    {    
      //BufferedImage bi = ImageIO.read(new File("1.pcoraw"));
        TIFFImageReaderSpi spi = new TIFFImageReaderSpi();
        URL resource = getClass().getResource("/test5.tif");
        ImageInputStream iis = ImageIO.createImageInputStream(resource.openConnection().getInputStream());
        TIFFImageReader reader = (TIFFImageReader)spi.createReaderInstance();        
        reader.setInput(iis);
        
//        TIFFImageReadParam tp = (TIFFImageReadParam)reader.getDefaultReadParam();
//        tp.setReadUnknownTags(true);
        //TODO BufferedImage bi = reader.read(0, read);
        BufferedImage bi = reader.read(0, reader.getDefaultReadParam());
        IIOMetadata meta = reader.getImageMetadata(0);
        TIFFImageMetadata tm = (TIFFImageMetadata)meta;
        
        B16ImageWriter writer = new B16ImageWriter();
        ImageTypeSpecifier imageType = reader.getImageTypes(0).next();
        PcoIIOMetadata convertedMetadata = (PcoIIOMetadata)writer.convertImageMetadata(tm, imageType, null);
        
        //values from the beginning of the pco tag in TIFF according to Hex-Editor
        BildStruct mBean = convertedMetadata.getBildStruct();
        assertEquals(2018, mBean.time.year);
        assertEquals(10, mBean.time.month);
        assertEquals(2, mBean.time.dayOfWeek);
        assertEquals(23, mBean.time.day);
        assertEquals(13, mBean.time.hour);
        assertEquals(5, mBean.time.minute);
        assertEquals(0, mBean.time.second);
        assertEquals(799, mBean.time.milliseconds);
        assertEquals(0, mBean.ticks);
        assertEquals(4096, mBean.xRes);
        assertEquals(2160, mBean.yRes);
        // values from the end
        assertEquals(1, mBean.binningX);
        assertEquals(1, mBean.binningY);
        assertEquals(TriggerMode.AUTOTRIGGER, mBean.triggerMode);
        assertEquals(CameraSyncMode.STANDALONE, mBean.cameraSyncMode);
        assertEquals(55_000_000, mBean.sensorReadoutFrequency);
        assertEquals(12_900_335, mBean.cameraSerialNo);
        assertEquals(SyncStatus.OFF, mBean.syncStatus);
        assertEquals(ImageType.UNDEFINED, mBean.imageType);
    }
}
