package de.pco.common;

/*-
 * #%L
 * pco-common
 * %%
 * Copyright (C) 2020 PCO
 * %%
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 * #L%
 */

import java.nio.ByteBuffer;

/**
 * Class represents the array of unsigned shorts.
 * @author PCO
 *
 */
public class UShortArray extends AbstractUnsignedArray 
{
    public static final int ELEMENT_MAX_VALUE = 65535;
    
    /**
     * Wraps the array of (signed) shorts that are to be interpreted as the unsigned short values 
     * @param shortArray
     */
    public UShortArray(short[] shortArray)
    {
        super(shortArray);
        if (shortArray == null) {
            throw new IllegalArgumentException("short array can not be null");
        }
        buffer = ByteBuffer.allocate(shortArray.length * 2);
        buffer.asShortBuffer().put(shortArray);
    }
    
    /**
     * Constructs the class from the array of the real unsigned short values, stored in a int array.
     * Performs also check of the value range. 
     * @param intArray
     * @throws IllegalArgumentException
     */
    public UShortArray(int[] intArray) throws IllegalArgumentException
    {
        super(intArray);
        if (intArray == null) {
            throw new IllegalArgumentException("int array can not be null");
        }
        for (int i = 0; i < intArray.length; i++) 
        {
            if ((intArray[i] < 0) || (intArray[i] > ELEMENT_MAX_VALUE)) {
                throw new IllegalArgumentException("Array contains values larger than " + ELEMENT_MAX_VALUE);
            }
        }
        short[] shortArray = new short[intArray.length];
        for (int i = 0; i < intArray.length; i++) 
        {
            shortArray[i] = (short)intArray[i];
        }
        buffer = ByteBuffer.allocate(shortArray.length * 2);
        buffer.asShortBuffer().put(shortArray);
    }
    
    /**
     * Converts the backing array of signed values into an array of unsigned ones and returns them as 
     * the int array.
     */
    @Override
    public int[] getArray() 
    {
        short[] backingArray = getBackingArray();
        int[] array = new int[backingArray.length];
        for (int i = 0; i < array.length; i++) {
            array[i] = Short.toUnsignedInt(backingArray[i]);
        }
        return array;
    }

    /**
     * Returns the array of the (signed) short values stored in the buffer (each value is taken from two bytes). 
     * This method is to be called in case of ImageIO framework to read the BufferedImage.  
     */
    @Override
    public short[] getBackingArray() 
    {
        short[] retArray = new short[buffer.capacity() / 2];
        buffer.asShortBuffer().get(retArray);
        return retArray;
    }
    
    /**
     * Returns the number of values stored.
     */
    @Override
    public int length() {
        return (this.buffer.capacity() / 2);
    }
}
