package de.pco.common;

/*-
 * #%L
 * pco-common
 * %%
 * Copyright (C) 2020 PCO
 * %%
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 * #L%
 */

import de.pco.sdk.PcoMetadataStruct;
import de.pco.sdk.Timestamp;

/**
 * Presents the container of the subsidiary info to the recorded image.
 * Encapsulates the recorder image number, PCO_METADATA_STRUCT and PCO_TIMESTAMP_STRUCT
 * @author PCO
 *
 */
public class ImageMetadata 
{
    private long recorderImageNumber;
    private PcoMetadataStruct pcoMetadataStruct;
    private Timestamp timestamp;
    
    /**
     * Constructor as in pco.recorder PCO_RecorderCopyImage function
     * @param recorderImageNumber
     * @param pcoMetadataStruct
     * @param timestamp
     */
    public ImageMetadata(long recorderImageNumber, PcoMetadataStruct pcoMetadataStruct, Timestamp timestamp)
    {
        this.recorderImageNumber = recorderImageNumber;
        this.pcoMetadataStruct = pcoMetadataStruct;
        this.timestamp = timestamp;
    }
    
    /**
     * Serial number of the camera
     * @return
     */
    public long getCameraSerialNo() 
    {
        if (pcoMetadataStruct != null) {
            return pcoMetadataStruct.getCameraSerialNo();
        }
        else {
            return -1;
        }
    }
    
    /**
     * Returns the number of the image returned by the FW of the camera
     * @return
     */
    public long getCameraImageNumber() 
    {
        if (timestamp != null) {
            return timestamp.imgCounter;
        }
        else if (pcoMetadataStruct != null) {
            return pcoMetadataStruct.getImageCounter();
        }
        else {
            return -1;
        }
    }

    /**
     * Returns the number of the image in the current record
     * @return
     */
    public long getRecorderImageNumber() {
        return recorderImageNumber;
    }
    
    /**
     * Returns the width of the image
     * @return
     */
    public int getSizeX() 
    {
        if (pcoMetadataStruct != null) {
            return pcoMetadataStruct.getImageSizeX();
        }
        else {
            return -1;
        }
    }
    
    /**
     * Returns the height of the image
     * @return
     */
    public int getSizeY() 
    {
        if (pcoMetadataStruct != null) {
            return pcoMetadataStruct.getImageSizeY();
        }
        else {
            return -1;
        }
    }
    
    /**
     * Returns the horizontal binning
     * @return
     */
    public byte getBinningX() 
    {
        if (pcoMetadataStruct != null) {
            return pcoMetadataStruct.getBinningX();
        }
        else {
            return -1;
        }
    }
    
    /**
     * Returns the vertical binning
     * @return
     */
    public byte getBinningY() 
    {
        if (pcoMetadataStruct != null) {
            return pcoMetadataStruct.getBinningY();
        }
        else {
            return -1;
        }
    }
    
    /**
     * Conversion factor of the sensor
     * @return
     */
    public int getSensorConvFactor() 
    {
        if (pcoMetadataStruct != null) {
            return pcoMetadataStruct.getSensorConvFactor();
        }
        else {
            return -1;
        }
    }
    
    /**
     * Returns the pixel clock
     * @return
     */
    public long getFramerateMilliHz() 
    {
        if (pcoMetadataStruct != null) {
            return pcoMetadataStruct.getFramerateMilliHz();
        }
        else {
            return -1;
        }
    }
    
    /**
     * Returns the current temperature of the sensor
     * @return
     */
    public short getSensorTemperature() 
    {
        if (pcoMetadataStruct != null) {
            return pcoMetadataStruct.getSensorTemperature();
        }
        else {
            return -1;
        }
    }

    /**
     * Returns the info from the PCO_METADATA_STRUCT
     * @return
     */
    public PcoMetadataStruct getPcoMetadataStruct() {
        return pcoMetadataStruct;
    }

    /**
     * Returns the timestamp
     * @return
     */
    public Timestamp getTimestamp() {
        return timestamp;
    }


    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((pcoMetadataStruct == null) ? 0 : pcoMetadataStruct.hashCode());
        result = prime * result + (int) (recorderImageNumber ^ (recorderImageNumber >>> 32));
        result = prime * result + ((timestamp == null) ? 0 : timestamp.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        ImageMetadata other = (ImageMetadata) obj;
        if (pcoMetadataStruct == null) {
            if (other.pcoMetadataStruct != null)
                return false;
        } else if (!pcoMetadataStruct.equals(other.pcoMetadataStruct))
            return false;
        if (recorderImageNumber != other.recorderImageNumber)
            return false;
        if (timestamp == null) {
            if (other.timestamp != null)
                return false;
        } else if (!timestamp.equals(other.timestamp))
            return false;
        return true;
    }

    @Override
    public String toString() {
        return "ImageMetadata [recorderImageNumber=" + recorderImageNumber + ", pcoMetadataStruct=" + pcoMetadataStruct
                + ", timestamp=" + timestamp + "]";
    }

    
}
