package de.pco.common.enums;

/*-
 * #%L
 * pco-common
 * %%
 * Copyright (C) 2020 PCO
 * %%
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 * #L%
 */

import java.util.Arrays;
import java.util.List;

import de.pco.common.MetadataBean;

/**
 * Enumerator of the fields of the MetadataBean class (Bild-Struct in SDK).
 * Contains info about he min and max values of the numeric fields.  
 * @author PCO
 *
 */
public enum MetadataField 
{
    TIME_YEAR,
    TIME_MONTH,
    TIME_DAY_OF_WEEK,
    TIME_DAY,
    TIME_HOUR,
    TIME_MINUTE,
    TIME_SECOND,
    TIME_MILLISECONDS,
    TICKS,
    X_RES,
    Y_RES,
    TEXT,
    IS_DOUBLE,
    THIS_IS_T0,
    WE_HAVE_T0,
    BW_MIN,
    BW_MAX,
    BW_LUT,
    R_MIN,
    R_MAX,
    G_MIN,
    G_MAX,
    B_MIN,
    B_MAX,
    COL_LUT,
    IS_COLOR,
    VERSION,
    BW_MIN2,
    BW_MAX2,
    BW_LUT2,
    R_MIN2,
    R_MAX2,
    G_MIN2,
    G_MAX2,
    B_MIN2,
    B_MAX2,
    COL_LUT2,
    ALIGN_UPPER,
    GAMMA_LUT,
    GAMMA_LUT_C,
    GAMMA_LUT2,
    GAMMA_LUT_C2,
    COLOR_PATTERN_TYPE,
    BIT_RES,
    D_SATURATION,
    I_SATURATION,
    VIBRANCE,
    COLOR_TEMP,
    COLOR_TINT,
    CONTRAST,
    GAMMA,
    COLOR_COEFF_A11,
    COLOR_COEFF_A12,
    COLOR_COEFF_A13,
    COLOR_COEFF_A21,
    COLOR_COEFF_A22,
    COLOR_COEFF_A23,
    COLOR_COEFF_A31,
    COLOR_COEFF_A32,
    COLOR_COEFF_A33,
    TIMESTAMP_POSITION,
    HAS_METADATA,
    SENSOR_CONV_FACTOR,
    CAMERA_TYPE,
    DARK_OFFSET,
    IMAGE_COUNTER,
    IMAGE_TIME_US,
    IMAGE_TIME_STATUS,
    EXPOSURE_TIMEBASE,
    SENSOR_TEMPERATURE,
    EXPOSURE_TIME,
    FRAMERATE_MILLI_HZ,
    BINNING_X,
    BINNING_Y,
    TRIGGER_MODE,
    CAMERA_SYNC_MODE,
    SENSOR_READOUT_FREQUENCY,
    CAMERA_SERIAL_NO,
    SYNC_STATUS,
    IMAGE_TYPE,
    COLOR_PATTERN,
    CAMERA_SUBTYPE,
    EVENT_NUMBER,
    IMAGE_SIZE_X_OFFSET,
    IMAGE_SIZE_Y_OFFSET,
    COMPRESSION,
    COMPRESSION_VERSION,
    PREDICTOR_START,
    RANDOM_START,
    RANDOM_INDEX,
    DIFF_TO_T0;
    
    private static List<MetadataField> booleanFields = Arrays.asList(new MetadataField[] {
        IS_DOUBLE, THIS_IS_T0, 
        WE_HAVE_T0, IS_COLOR,
        ALIGN_UPPER, HAS_METADATA
    });
    
    private static List<MetadataField> enumFields = Arrays.asList(new MetadataField[] {
        VERSION, TIMESTAMP_POSITION,
        COLOR_PATTERN_TYPE, CAMERA_TYPE,
        IMAGE_TIME_STATUS, TRIGGER_MODE,
        CAMERA_SYNC_MODE, SYNC_STATUS,
        IMAGE_TYPE, CAMERA_SUBTYPE, 
        EXPOSURE_TIMEBASE                
    });
    
    private static List<MetadataField> wordMetadataFields = Arrays.asList(new MetadataField[] { 
        TIME_YEAR, TIME_MILLISECONDS,
        X_RES, Y_RES,
        BW_MIN, BW_MAX,
        BW_MIN2, BW_MAX2,
        R_MIN, R_MAX,
        R_MIN2, R_MAX2,
        G_MIN, G_MAX,
        G_MIN2, G_MAX2,
        B_MIN, B_MAX,
        B_MIN2, B_MAX2,
        SENSOR_CONV_FACTOR, DARK_OFFSET,
        COLOR_PATTERN, IMAGE_SIZE_X_OFFSET,
        IMAGE_SIZE_Y_OFFSET
    });
    
    private static List<MetadataField> uIntMetadataFields = Arrays.asList(new MetadataField[] { // are DW
        TICKS, // although it is int
        IMAGE_COUNTER, IMAGE_TIME_US, 
        EXPOSURE_TIME, FRAMERATE_MILLI_HZ,
        SENSOR_READOUT_FREQUENCY, CAMERA_SERIAL_NO, 
        EVENT_NUMBER, COMPRESSION, 
        COMPRESSION_VERSION,  
        RANDOM_START
    });
    
    private static List<MetadataField> zeroMetadataFields = Arrays.asList(new MetadataField[] {
        PREDICTOR_START, RANDOM_INDEX,
        COL_LUT, COL_LUT2,
        BW_LUT, BW_LUT2
    });
    
    private static List<MetadataField> int100MetadataFields = Arrays.asList(new MetadataField[] {
        I_SATURATION, 
        VIBRANCE, COLOR_TINT,
        CONTRAST
    });
       
    private static List<MetadataField> double40MetadataFields = Arrays.asList(new MetadataField[] {
        GAMMA_LUT, GAMMA_LUT2,
        GAMMA_LUT_C, GAMMA_LUT_C2,
    });
    
    private static List<MetadataField> double3MetadataFields = Arrays.asList(new MetadataField[] {
        COLOR_COEFF_A11,
        COLOR_COEFF_A12, COLOR_COEFF_A13,
        COLOR_COEFF_A21, COLOR_COEFF_A22,
        COLOR_COEFF_A23, COLOR_COEFF_A31,
        COLOR_COEFF_A32, COLOR_COEFF_A33
    });
    
    private static List<MetadataField> specialMetadataFields = Arrays.asList(new MetadataField[] {
        TEXT,
        BIT_RES, COLOR_TEMP, 
        BINNING_X, BINNING_Y,
        DIFF_TO_T0, SENSOR_TEMPERATURE,
        TIME_MONTH, TIME_DAY_OF_WEEK, TIME_DAY,
        TIME_HOUR, TIME_MINUTE,
        TIME_SECOND, D_SATURATION, GAMMA
    });
    
    /**
     * Returns the minimal falue of a numeric field from the MetadataBean
     * @param enumValue
     * @return
     */
    public static Comparable getMinValue(MetadataField enumValue) 
    {
        if (   wordMetadataFields.contains(enumValue)
            || uIntMetadataFields.contains(enumValue)
            || zeroMetadataFields.contains(enumValue)) 
        {
            return 0;
        }
        else if (int100MetadataFields.contains(enumValue)) {
            return -100;
        }
        else if (double40MetadataFields.contains(enumValue)) {
            return 40.0;
        }
        else if (double3MetadataFields.contains(enumValue)) {
            return -3.0;
        }
        else if (specialMetadataFields.contains(enumValue)) 
        {
            switch (enumValue) {
            case BIT_RES:
                return 4;  // although B16 has minimum 12 bits, TIFF can be of 4 bits
            case BINNING_X:
            case BINNING_Y:
                return 1;
            case DIFF_TO_T0:
                return Integer.MIN_VALUE;
            case SENSOR_TEMPERATURE:
                return Short.MIN_VALUE;
            case TIME_MONTH:
            case TIME_DAY_OF_WEEK:
            case TIME_DAY:
            case TIME_HOUR:
            case TIME_MINUTE:
            case TIME_SECOND:
            case TIME_MILLISECONDS:
                return 0;
            case COLOR_TEMP:
                return 3500;
            case D_SATURATION:
                return -100.0;
            case GAMMA:
                return 40;
            }
        }
        
        return null;
    }
    
    /**
     * Returns the maximal falue of a numeric field from the MetadataBean
     * @param enumValue
     * @return
     */
    public static Comparable getMaxValue(MetadataField enumValue) 
    {
        if (wordMetadataFields.contains(enumValue)) {
            return 65535;
        }
        else if (uIntMetadataFields.contains(enumValue)) {
            return Integer.MAX_VALUE;
        }
        else if (zeroMetadataFields.contains(enumValue)) {
            return 0;
        }
        else if (int100MetadataFields.contains(enumValue)) {
            return 100;
        }
        else if (double40MetadataFields.contains(enumValue)) {
            return 250.0;
        }
        else if (double3MetadataFields.contains(enumValue)) {
            return 3.0;
        }
        else if (specialMetadataFields.contains(enumValue)) 
        {
            switch (enumValue) {
            case BIT_RES:
                return 32;
            case BINNING_X:
            case BINNING_Y:
                return 32;
            case DIFF_TO_T0:
                return Integer.MAX_VALUE;
            case SENSOR_TEMPERATURE:
                return Short.MAX_VALUE;
            case TIME_MONTH:
                return 11;
            case TIME_DAY_OF_WEEK:
                return 6;
            case TIME_DAY:
                return 30;
            case TIME_HOUR:
                return 23;
            case TIME_MINUTE:
            case TIME_SECOND:
                return 59;                
            case TIME_MILLISECONDS:
                return 999;
            case COLOR_TEMP:
                return 20000;
            case D_SATURATION:
                return 100.0;
            case GAMMA:
                return 250;
            }
        }
        
        return null;
    }
    
    /**
     * Returns the default falue of a numeric field from the MetadataBean
     * @param enumValue
     * @return
     */
    public static Object getDefaultValue(MetadataField enumValue) 
    {
        if (   wordMetadataFields.contains(enumValue)
            || uIntMetadataFields.contains(enumValue)
            || zeroMetadataFields.contains(enumValue)
            || int100MetadataFields.contains(enumValue)
            )
        {
            return 0;
        }
        else if (double3MetadataFields.contains(enumValue)) 
        {
            return 0.0;
        }
        else if (double40MetadataFields.contains(enumValue)) 
        {
            return 40.0;
        }
        else if (specialMetadataFields.contains(enumValue)) 
        {
            switch (enumValue) {
            case BIT_RES:
                return 16;
            case BINNING_X:
            case BINNING_Y:
                return 1;
            case DIFF_TO_T0:
                return 0;
            case SENSOR_TEMPERATURE:
                return Short.MIN_VALUE;
            case TIME_MONTH:
            case TIME_DAY_OF_WEEK:
            case TIME_DAY:
            case TIME_HOUR:
            case TIME_MINUTE:
            case TIME_SECOND:
            case TIME_MILLISECONDS:
                return 0;
            case COLOR_TEMP:
                return 3500;
            case D_SATURATION:
                return 0.0;
            case GAMMA:
                return 40;
            }
        }
        
        return null;
    }
    
    /**
     * Returns the step value for the particular numeric field
     * @param enumValue
     * @return
     */
    public static Object getStepValue(MetadataField enumValue) 
    {
        if (double40MetadataFields.contains(enumValue)
             || double3MetadataFields.contains(enumValue)
             || enumValue == D_SATURATION) 
        {
            return 0.1d;
        }
        else {
            return 1;
        }
    }
    
    /**
     * Returns the possible values of the field from the MetadataBean that is an enumerator 
     * @param enumValue
     * @return
     */
    public static Object[] getEnumFieldValues(MetadataField enumValue) 
    {
        switch(enumValue) 
        {
        case VERSION:
            return B16Version.values();
        case TIMESTAMP_POSITION:
            return TimestampPosition.values();
        case COLOR_PATTERN_TYPE:
            return ColorPatternType.values();
        case CAMERA_TYPE:
            return CameraType.values();
        case IMAGE_TIME_STATUS:
            return ImageTimeStatus.values();
        case TRIGGER_MODE:
            return TriggerMode.values();
        case CAMERA_SYNC_MODE:
            return CameraSyncMode.values();
        case SYNC_STATUS:
            return SyncStatus.values();
        case IMAGE_TYPE:
            return ImageType.values();
        case CAMERA_SUBTYPE:
            return CameraSubtype.values();
        case EXPOSURE_TIMEBASE:
            return Timebase.values();
        default:
            return null;
        }
    }
    
    /**
     * Returns whether the parameter is a numeric field of the MetadataBean.
     * @param enumValue
     * @return
     */
    public static boolean isNumericField(MetadataField enumValue) 
    {
        if (    booleanFields.contains(enumValue) 
            ||  enumFields.contains(enumValue)
            ||  enumValue == MetadataField.TEXT)
        {
            return false;
        }
        else {
            return true;
        }
    }
    
    /**
     * Returns whether the parameter is a textual field of the MetadataBean.
     * @param enumValue
     * @return
     */
    public static boolean isTextField(MetadataField enumValue)
    {
        if (enumValue == MetadataField.TEXT) {
            return true;
        }
        else {
            return false;
        }
    }
    
    /**
     * Returns whether the parameter is an enumerator field of the MetadataBean.
     * @param enumValue
     * @return
     */
    public static boolean isEnumField(MetadataField enumValue)
    {
        if (enumFields.contains(enumValue)) {
            return true;
        }
        else {
            return false;
        }
    }
    
    /**
     * Returns whether the parameter is a boolean field of the MetadataBean.
     * @param enumValue
     * @return
     */
    public static boolean isBooleanField(MetadataField enumValue)
    {
        if (booleanFields.contains(enumValue)) {
            return true;
        }
        else {
            return false;
        }
    }
    
    /**
     * By giving the particular field and the object of the MetadataBean class, this method
     * returns the value of this field in this object.
     * @param enumValue
     * @return
     */
    public static Object getValueOfField(MetadataBean metadata, MetadataField enumValue) 
    {
        switch (enumValue) {
        case TIME_YEAR: return metadata.time.year;
        case TIME_MONTH: return metadata.time.month;
        case TIME_DAY_OF_WEEK: return metadata.time.dayOfWeek;
        case TIME_DAY: return metadata.time.day;
        case TIME_HOUR: return metadata.time.hour;
        case TIME_MINUTE: return metadata.time.minute;
        case TIME_SECOND: return metadata.time.second;
        case TIME_MILLISECONDS: return metadata.time.milliseconds;
        case TICKS: return metadata.ticks;  
        case X_RES: return metadata.xRes;
        case Y_RES: return metadata.yRes;
        case TEXT: return metadata.text;
        case IS_DOUBLE: return metadata.isDouble;
        case THIS_IS_T0: return metadata.thisIsT0;
        case WE_HAVE_T0: return metadata.weHaveT0;
        case BW_MIN: return metadata.bwMin;
        case BW_MAX: return metadata.bwMax;
        case BW_LUT: return metadata.bwLut;
        case R_MIN: return metadata.rMin;
        case R_MAX: return metadata.rMax;
        case G_MIN: return metadata.gMin;
        case G_MAX: return metadata.gMax;
        case B_MIN: return metadata.bMin;
        case B_MAX: return metadata.bMax;
        case COL_LUT: return metadata.colLut;
        case IS_COLOR: return metadata.isColor;
        case VERSION: return metadata.version;
        case BW_MIN2: return metadata.bwMin2;
        case BW_MAX2: return metadata.bwMax2;
        case BW_LUT2: return metadata.bwLut2;
        case R_MIN2: return metadata.rMin2;
        case R_MAX2: return metadata.rMax2;
        case G_MIN2: return metadata.gMin2;
        case G_MAX2: return metadata.gMax2;
        case B_MIN2: return metadata.bMin2;
        case B_MAX2: return metadata.bMax2;
        case COL_LUT2: return metadata.colLut;
        case ALIGN_UPPER: return metadata.alignUpper;
        case GAMMA_LUT: return metadata.gammaLut;
        case GAMMA_LUT_C: return metadata.gammaLutC;
        case GAMMA_LUT2: return metadata.gammaLut2;
        case GAMMA_LUT_C2: return metadata.gammaLutC2;
        case COLOR_PATTERN_TYPE: return metadata.colorPatternType;
        case BIT_RES: return metadata.bitRes;
        case D_SATURATION: return metadata.dSaturation;
        case I_SATURATION: return metadata.iSaturation;
        case VIBRANCE: return metadata.vibrance;
        case COLOR_TEMP: return metadata.colorTemp;
        case COLOR_TINT: return metadata.colorTint;
        case CONTRAST: return metadata.contrast;
        case GAMMA: return metadata.gamma;
        case COLOR_COEFF_A11: return metadata.colorCoeff.a11;
        case COLOR_COEFF_A12: return metadata.colorCoeff.a12;
        case COLOR_COEFF_A13: return metadata.colorCoeff.a13;
        case COLOR_COEFF_A21: return metadata.colorCoeff.a21;
        case COLOR_COEFF_A22: return metadata.colorCoeff.a22;
        case COLOR_COEFF_A23: return metadata.colorCoeff.a23;
        case COLOR_COEFF_A31: return metadata.colorCoeff.a31;
        case COLOR_COEFF_A32: return metadata.colorCoeff.a32;
        case COLOR_COEFF_A33: return metadata.colorCoeff.a33;
        case TIMESTAMP_POSITION: return metadata.timestampPosition;
        case HAS_METADATA: return metadata.hasMetaData;
        case CAMERA_TYPE: return metadata.cameraType;
        case SENSOR_CONV_FACTOR: return metadata.sensorConvFactor;
        case DARK_OFFSET: return metadata.darkOffset;
        case IMAGE_COUNTER: return metadata.imageCounter;
        case IMAGE_TIME_US: return metadata.imageTimeUs;
        case IMAGE_TIME_STATUS: return metadata.imageTimeStatus;
        case EXPOSURE_TIMEBASE: return metadata.exposureTimebase;
        case SENSOR_TEMPERATURE: return metadata.sensorTemperature;
        case EXPOSURE_TIME: return metadata.exposureTime;
        case FRAMERATE_MILLI_HZ: return metadata.framerateMilliHz;
        case BINNING_X: return metadata.binningX;
        case BINNING_Y: return metadata.binningY;
        case TRIGGER_MODE: return metadata.triggerMode;
        case CAMERA_SYNC_MODE: return metadata.cameraSyncMode;
        case SENSOR_READOUT_FREQUENCY: return metadata.sensorReadoutFrequency;
        case CAMERA_SERIAL_NO: return metadata.cameraSerialNo;
        case SYNC_STATUS: return metadata.syncStatus;
        case IMAGE_TYPE: return metadata.imageType;
        case COLOR_PATTERN: return metadata.colorPattern;
        case CAMERA_SUBTYPE: return metadata.cameraSubtype;
        case EVENT_NUMBER: return metadata.eventNumber;
        case IMAGE_SIZE_X_OFFSET: return metadata.imageSizeXoffset;
        case IMAGE_SIZE_Y_OFFSET: return metadata.imageSizeYoffset;
        case COMPRESSION: return metadata.compression;
        case COMPRESSION_VERSION: return metadata.compressionVersion;
        case PREDICTOR_START: return metadata.predictorStart;
        case RANDOM_START: return metadata.randomStart;
        case RANDOM_INDEX: return metadata.randomIndex;
        case DIFF_TO_T0: return metadata.diffToT0;
        default: throw new IllegalArgumentException("illegal MetadataField constant");
        }
    }
    
    /**
     * Sets the value
     * @param metadata
     * @param field
     * @param value
     */
    public static void setValueOfField(MetadataBean metadata, MetadataField field, Object value) 
    {
        switch (field) {
        case TIME_YEAR: 
            metadata.time.year = (int)value;
            break;
        case TIME_MONTH: 
            metadata.time.month = (int)value;
            break;
        case TIME_DAY_OF_WEEK: 
            metadata.time.dayOfWeek = (int)value;
            break;
        case TIME_DAY: 
            metadata.time.day = (int)value;
            break;
        case TIME_HOUR: 
            metadata.time.hour = (int)value;
            break;
        case TIME_MINUTE: 
            metadata.time.minute = (int)value;
            break;
        case TIME_SECOND: 
            metadata.time.second = (int)value;
            break;
        case TIME_MILLISECONDS: 
            metadata.time.milliseconds = (int)value;
            break;
        case TICKS: 
            metadata.ticks = (int)value;
            break;
        case X_RES: 
            metadata.xRes = (int)value;
            break;
        case Y_RES: 
            metadata.yRes = (int)value;
            break;
        case TEXT: 
            metadata.text = (String)value;
            break;
        case IS_DOUBLE: 
            metadata.isDouble = (boolean)value;
            break;
        case THIS_IS_T0: 
            metadata.thisIsT0 = (boolean)value;
            break;
        case WE_HAVE_T0: 
            metadata.weHaveT0 = (boolean)value;
            break;
        case BW_MIN: 
            metadata.bwMin = (int)value;
            break;
        case BW_MAX: 
            metadata.bwMax = (int)value;
            break;
        case BW_LUT: 
            metadata.bwLut = (int)value;
            break;
        case R_MIN: 
            metadata.rMin = (int)value;
            break;
        case R_MAX: 
            metadata.rMax = (int)value;
            break;
        case G_MIN: 
            metadata.gMin = (int)value;
            break;
        case G_MAX: 
            metadata.gMax = (int)value;
            break;
        case B_MIN: 
            metadata.bMin = (int)value;
            break;
        case B_MAX: 
            metadata.bMax = (int)value;
            break;
        case COL_LUT: 
            metadata.colLut = (int)value;
            break;
        case IS_COLOR: 
            metadata.isColor = (boolean)value;
            break;
        case VERSION: 
            metadata.version = (B16Version)value;
            break;
        case BW_MIN2: 
            metadata.bwMin2 = (int)value;
            break;
        case BW_MAX2: 
            metadata.bwMax2 = (int)value;
            break;
        case BW_LUT2: 
            metadata.bwLut2 = (int)value;
            break;
        case R_MIN2: 
            metadata.rMin2 = (int)value;
            break;
        case R_MAX2: 
            metadata.rMax2 = (int)value;
            break;
        case G_MIN2: 
            metadata.gMin2 = (int)value;
            break;
        case G_MAX2: 
            metadata.gMax2 = (int)value;
            break;
        case B_MIN2: 
            metadata.bMin2 = (int)value;
            break;
        case B_MAX2: 
            metadata.bMax2 = (int)value;
            break;
        case COL_LUT2: 
            metadata.colLut = (int)value;
            break;
        case ALIGN_UPPER: 
            metadata.alignUpper = (boolean)value;
            break;
        case GAMMA_LUT: 
            metadata.gammaLut = (double)value;
            break;
        case GAMMA_LUT_C: 
            metadata.gammaLutC = (double)value;
            break;
        case GAMMA_LUT2: 
            metadata.gammaLut2 = (double)value;
            break;
        case GAMMA_LUT_C2: 
            metadata.gammaLutC2 = (double)value;
            break;
        case COLOR_PATTERN_TYPE: 
            metadata.colorPatternType = (ColorPatternType)value;
            break;
        case BIT_RES: 
            metadata.bitRes = (int)value;
            break;
        case D_SATURATION: 
            metadata.dSaturation = (double)value;
            break;
        case I_SATURATION: 
            metadata.iSaturation = (int)value;
            break;
        case VIBRANCE: 
            metadata.vibrance = (int)value;
            break;
        case COLOR_TEMP: 
            metadata.colorTemp = (int)value;
            break;
        case COLOR_TINT: 
            metadata.colorTint = (int)value;
            break;
        case CONTRAST: 
            metadata.contrast = (int)value;
            break;
        case GAMMA: 
            metadata.gamma = (int)value;
            break;
        case COLOR_COEFF_A11: 
            metadata.colorCoeff.a11 = (double)value;
            break;
        case COLOR_COEFF_A12: 
            metadata.colorCoeff.a12 = (double)value;
            break;
        case COLOR_COEFF_A13: 
            metadata.colorCoeff.a13 = (double)value;
            break;
        case COLOR_COEFF_A21: 
            metadata.colorCoeff.a21 = (double)value;
            break;
        case COLOR_COEFF_A22: 
            metadata.colorCoeff.a22 = (double)value;
            break;
        case COLOR_COEFF_A23:
            metadata.colorCoeff.a23 = (double)value;
            break;
        case COLOR_COEFF_A31: 
            metadata.colorCoeff.a31 = (double)value;
            break;
        case COLOR_COEFF_A32: 
            metadata.colorCoeff.a32 = (double)value;
            break;
        case COLOR_COEFF_A33: 
            metadata.colorCoeff.a33 = (double)value;
            break;
        case TIMESTAMP_POSITION: 
            metadata.timestampPosition = (TimestampPosition)value;
            break;
        case HAS_METADATA: 
            metadata.hasMetaData = (boolean)value;
            break;
        case CAMERA_TYPE: 
            metadata.cameraType = (CameraType)value;
            break;
        case SENSOR_CONV_FACTOR: 
            metadata.sensorConvFactor = (int)value;
            break;
        case DARK_OFFSET: 
            metadata.darkOffset = (int)value;
            break;
        case IMAGE_COUNTER: 
            metadata.imageCounter = (int)value;
            break;
        case IMAGE_TIME_US: 
            metadata.imageTimeUs = (int)value;
            break;
        case IMAGE_TIME_STATUS: 
            metadata.imageTimeStatus = (ImageTimeStatus)value;
            break;
        case EXPOSURE_TIMEBASE: 
            metadata.exposureTimebase = (Timebase)value;
            break;
        case SENSOR_TEMPERATURE: 
            metadata.sensorTemperature = (short)((int)value);
            break;
        case EXPOSURE_TIME: 
            metadata.exposureTime = (int)value;
            break;
        case FRAMERATE_MILLI_HZ: 
            metadata.framerateMilliHz = (int)value;
            break;
        case BINNING_X: 
            metadata.binningX = (byte)((int)value);
            break;
        case BINNING_Y: 
            metadata.binningY = (byte)((int)value);
            break;
        case TRIGGER_MODE: 
            metadata.triggerMode = (TriggerMode)value;
            break;
        case CAMERA_SYNC_MODE: 
            metadata.cameraSyncMode = (CameraSyncMode)value;
            break;
        case SENSOR_READOUT_FREQUENCY: 
            metadata.sensorReadoutFrequency = (int)value;// from long
            break;
        case CAMERA_SERIAL_NO: 
            metadata.cameraSerialNo = (int)value;// from long
            break;
        case SYNC_STATUS: 
            metadata.syncStatus = (SyncStatus)value;
            break;
        case IMAGE_TYPE: 
            metadata.imageType = (ImageType)value;
            break;
        case COLOR_PATTERN: 
            metadata.colorPattern = (int)value;
            break;
        case CAMERA_SUBTYPE: 
            metadata.cameraSubtype = (CameraSubtype)value;
            break;
        case EVENT_NUMBER: 
            metadata.eventNumber = (int)value;// from long
            break;
        case IMAGE_SIZE_X_OFFSET: 
            metadata.imageSizeXoffset = (int)value;
            break;
        case IMAGE_SIZE_Y_OFFSET: 
            metadata.imageSizeYoffset = (int)value;
            break;
        case COMPRESSION: 
            metadata.compression =  (int)value;// from long
            break;
        case COMPRESSION_VERSION: 
            metadata.compressionVersion = (int)value;// from long
            break;
        case PREDICTOR_START: 
            metadata.predictorStart = (int)value; // from long
            break;
        case RANDOM_START: 
            metadata.randomStart = (int)value; // from long
            break;
        case RANDOM_INDEX: 
            metadata.randomIndex = (int)value; // from long
            break;
        case DIFF_TO_T0: 
            metadata.diffToT0 = (int)value;
            break;
        default: throw new IllegalArgumentException("illegal MetadataField constant");
        }
    }
}
