package de.pco.common.enums;

/*-
 * #%L
 * pco-common
 * %%
 * Copyright (C) 2020 PCO
 * %%
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 * #L%
 */

/**
 * Enumerator of the subtypes of the particular family of the cameras.
 * @author PCO
 *
 */
public enum CameraSubtype 
{
    UNDEFINED(0),    

 // DIMAX Family    
    PCO_DIMAX_Weisscam(0x0064),    // 100 = Weisscam, all features
    PCO_DIMAX_HD(0x80FF),          // pco.dimax HD
    PCO_DIMAX_HD_PLUS(0xC0FF),     // pco.dimax HD+
    PCO_DIMAX_X35(0x00C8),         // 200 = Weisscam/P+S HD35
    
    PCO_DIMAX_HS1(0x207F),
    PCO_DIMAX_HS2(0x217F),
    PCO_DIMAX_HS4(0x237F),
    
    PCO_DIMAX_CS_AM_DEPRECATED(0x407F),
    PCO_DIMAX_CS_1(0x417F),
    PCO_DIMAX_CS_2(0x427F),
    PCO_DIMAX_CS_3(0x437F),
    PCO_DIMAX_CS_4(0x447F),

 // PANDA Family
    PCO_PANDA_42(0x0000),        // pco.panda 4.2
    PCO_PANDA_42_BI(0x0001),     // pco.panda 4.2 bi
    PCO_PANDA_150(0x0002),       // pco.panda 15
    PCO_PANDA_OEM1(0x0003),

// EDGE Family
    PCO_EDGE_SPRINGFIELD(0x0006),
    PCO_EDGE_31(0x0031),
    PCO_EDGE_42(0x0042),
    PCO_EDGE_55(0x0055),
    PCO_EDGE_DEVELOPMENT(0x0100),
    PCO_EDGE_X2(0x0200),
    PCO_EDGE_RESOLFT(0x0300),
    PCO_EDGE_GOLD(0x0FF0),
    PCO_EDGE_DUAL_CLOCK(0x000D),
    PCO_EDGE_DICAM(0xDC00),
    PCO_EDGE_42_LT(0x8042),
    
    PCO_EDGE_42_BI(0x0001),       // pco.edge 4.2 bi
    PCO_EDGE_260(0x0002),         // pco.edge 26

// DICAM Family
    PCO_DICAM_C1(0x0001),         // pco.dicam C1
    PCO_DICAM_C2(0x0002),         // reserved
    PCO_DICAM_C3(0x0003),         // reserved
    PCO_DICAM_C4(0x0004);         // pco.dicam C4
    
    private int value;
    
    /**
     * Constructor
     * @param value
     */
    CameraSubtype(int value) {
        this.value = value;
    }
    
    /**
     * Convertor from the int value and the camera family type to the value of the enumerator.
     * Returns UNDEFINED when the second parameter is not the family name 
     * @param value
     * @param familyType
     * @return
     * @throws IllegalArgumentException
     */
    public static CameraSubtype valueOf(int value, CameraType familyType) throws IllegalArgumentException 
    {
        switch (familyType) 
        {
        case PCO_FAMILY_DIMAX:
            switch (value) 
            {
                case 0x0064 : return PCO_DIMAX_Weisscam;
                case 0x80FF : return PCO_DIMAX_HD;
                case 0xC0FF : return PCO_DIMAX_HD_PLUS;
                case 0x00C8 : return PCO_DIMAX_X35;
                case 0x207F : return PCO_DIMAX_HS1;
                case 0x217F : return PCO_DIMAX_HS2;
                case 0x237F : return PCO_DIMAX_HS4;
                case 0x407F : return PCO_DIMAX_CS_AM_DEPRECATED;
                
                case 0x417F : return PCO_DIMAX_CS_1;
                case 0x427F : return PCO_DIMAX_CS_2;
                case 0x437F : return PCO_DIMAX_CS_3;
                case 0x447F : return PCO_DIMAX_CS_4;
                default : throw new IllegalArgumentException("incorrect camera subtype constant number");
            }
        case PCO_FAMILY_EDGE:
            switch (value) 
            {
                case 0x0006 : return PCO_EDGE_SPRINGFIELD;
                case 0x0031 : return PCO_EDGE_31;
                case 0x0042 : return PCO_EDGE_42;
                case 0x0055 : return PCO_EDGE_55;
                case 0x0100 : return PCO_EDGE_DEVELOPMENT;
                case 0x0200 : return PCO_EDGE_X2;
                case 0x0300 : return PCO_EDGE_RESOLFT;
                case 0x0FF0 : return PCO_EDGE_GOLD;
                case 0x000D : return PCO_EDGE_DUAL_CLOCK;
                case 0xDC00 : return PCO_EDGE_DICAM;
                case 0x8042 : return PCO_EDGE_42_LT;
                
                case 0x0001 : return PCO_EDGE_42_BI;       // pco.edge 4.2 bi
                case 0x0002 : return PCO_EDGE_260;         // pco.edge 26
                default : throw new IllegalArgumentException("incorrect camera subtype constant number");
            }
        case PCO_FAMILY_PANDA:
            switch (value)
            {
                case 0x0000 : return PCO_PANDA_42;        // pco.panda 4.2
                case 0x0001 : return PCO_PANDA_42_BI;     // pco.panda 4.2 bi
                case 0x0002 : return PCO_PANDA_150;       // pco.panda 15
                case 0x0003 : return PCO_PANDA_OEM1;
                default : throw new IllegalArgumentException("incorrect camera subtype constant number");
            }
        case PCO_FAMILY_DICAM:
            switch (value)
            {
                case 0x0001 : return PCO_DICAM_C1;        // pco.dicam C1
                case 0x0002 : return PCO_DICAM_C2;        // reserved
                case 0x0003 : return PCO_DICAM_C3;        // reserved
                case 0x0004 : return PCO_DICAM_C4;        // pco.dicam C4
                default : throw new IllegalArgumentException("incorrect camera subtype constant number");  
            }
        default : return UNDEFINED;
        }
    }
    
    /**
     * Getter
     * @return
     */
    public int getValue() {
        return value;
    }
    
}
