package de.pco.common;

/*-
 * #%L
 * pco-common
 * %%
 * Copyright (C) 2020 PCO
 * %%
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 * #L%
 */

import java.nio.ByteBuffer;

/**
 * Class that represent the unsigned byte array.
 * @author PCO
 *
 */
public class UByteArray extends AbstractUnsignedArray 
{
    public static final int ELEMENT_MAX_VALUE = 255;
    
    /**
     * Loads the data from the (signed) byte array, where the particular byte values are 
     * to be interpreted as the unsigned values  
     * @param byteArray
     */
    public UByteArray(byte[] byteArray) 
    {
        super(byteArray);
        if (byteArray == null) {
            throw new IllegalArgumentException("byte array can not be null");
        }
        this.buffer = ByteBuffer.wrap(byteArray);
    }
    
    /**
     * Loads the data from the int array that should actually store the unsigned bytes. Checks whether 
     * the values are in the range.
     * @param intArray
     */
    public UByteArray(int[] intArray) throws IllegalArgumentException
    {
        super(intArray);
        if (intArray == null) {
            throw new IllegalArgumentException("int array can not be null");
        }
        for (int i = 0; i < intArray.length; i++) 
        {
            if ((intArray[i] < 0) || (intArray[i] > ELEMENT_MAX_VALUE)) {
                throw new IllegalArgumentException("Array contains values larger than " + ELEMENT_MAX_VALUE);
            }
        }
        byte[] byteArray = new byte[intArray.length];
        for (int i = 0; i < intArray.length; i++) 
        {
            byteArray[i] = (byte)intArray[i];
        }
        this.buffer = ByteBuffer.wrap(byteArray);
    }
    
    /**
     * Returns the array of the unsigned byte values in an int array
     */
    @Override
    public int[] getArray() 
    {
        byte[] backingArray = getBackingArray();
        int[] array = new int[backingArray.length];
        for (int i = 0; i < array.length; i++) {
            array[i] = Byte.toUnsignedInt(backingArray[i]);
        }
        return array;
    }

    /**
     * Returns the (signed) sequence of bytes from the buffer as an array. This method is to be used
     * in the ImageIO framework.
     */
    @Override
    public byte[] getBackingArray() {
        return this.buffer.array();
    }
    
    /**
     * Returns the number of values stored.
     */
    @Override
    public int length() {
        return this.buffer.capacity();
    }

}
