package de.pco.common;

/*-
 * #%L
 * pco-common
 * %%
 * Copyright (C) 2020 PCO
 * %%
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 * #L%
 */

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.nio.ByteBuffer;

/**
 * Abstract class that represents an array of values of an unsigned data type.
 * Due to the inabilities of Java, these are stored in the backing buffer as the
 * signed values of a particular Java data type and get converted to the array of 
 * unsigned values when {@link getArray()} is called.
 * @author PCO
 *
 */
public abstract class AbstractUnsignedArray 
{
    protected ByteBuffer buffer;
    

    /**
     * This constructor from the array of bytes or shorts will be used after the native call 
     * to record the data will be performed in the C level of JNI. 
     * @param signedSubtypeArray
     */
    protected AbstractUnsignedArray(Object signedSubtypeArray) {
    }

    /**
     * Constructor for creating the array from the real unsigned values. This would be used
     * after manual computation with the image values to wrap the array again. 
     * @param array
     * @throws IllegalArgumentException
     */
    protected AbstractUnsignedArray(int[] array) throws IllegalArgumentException {
    }
    
    /**
     * Returns the factual array of unsigned values. To be called when there is a need for displaying
     * or manually counting with the real values. Converts from the backing byte array every time.  
     * @return
     */
    public abstract int[] getArray();
    
    /**
     * Returns the array as it is stored in the buffer as the array of a particular Java data type. 
     * To be used for reading to a BufferedImage in the ImageIO framework. 
     * @return
     */
    public abstract Object getBackingArray();
    
    /**
     * Returns an InputStream on the backing buffer. 
     * @return
     */
    public InputStream getInputStream() {
        return new ByteArrayInputStream(buffer.array());
    }
    
    /**
     * Returns the number of stored values, i.e. the length of both arrays returned by {@link getArray()} 
     * and {@link getBackingArray()}.
     * @return
     */
    public abstract int length();
    
    //////////////////////////////////// GENERATED
    
    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((buffer == null) ? 0 : buffer.hashCode());
        return result;
    }
    
    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        AbstractUnsignedArray other = (AbstractUnsignedArray) obj;
        if (buffer == null) {
            if (other.buffer != null)
                return false;
        } else if (!buffer.equals(other.buffer))
            return false;
        return true;
    }
    
    @Override
    public String toString() {
        return "AbstractUnsignedArray [buffer=" + buffer + "]";
    }
}
