package de.pco.common;

/*-
 * #%L
 * pco-common
 * %%
 * Copyright (C) 2020 PCO
 * %%
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 * #L%
 */

import java.io.InputStream;

/**
 * Class represents the image and its metadata.
 * @author PCO
 *
 */
public abstract class AbstractImageData 
{
    private int width;
    private int height; 
    
    protected AbstractUnsignedArray image;
    private AbstractMetadata metadata;
    
    /**
     * Constructor from the wrapped array of values, without the metadata.
     * @param image
     * @param width
     * @param height
     */
    protected AbstractImageData(AbstractUnsignedArray image, int width, int height) 
    {
        if (image == null) {
            throw new IllegalArgumentException("image can not be null");
        }
        this.image = image;
        this.width = width;
        this.height = height;
    }
    
    /**
     * Constructor from the wrapped array of values and from the metadata which contains also width and height.
     * @param image
     * @param metadata
     */
    protected AbstractImageData(AbstractUnsignedArray image, AbstractMetadata metadata) 
    {
        if (image == null) {
            throw new IllegalArgumentException("image can not be null");
        }
        if (metadata == null) {
            throw new IllegalArgumentException("metadata can not be null");
        }
        this.image = image;
        this.metadata = metadata;
        this.width = metadata.getImageSizeX();
        this.height = metadata.getImageSizeY();
    }
    
    /**
     * Stub of the constructor from a recounted array of values.
     * Subclasses are expected to extend this constructor to set the member 
     * field of the image values by performing a check of the array values 
     * to fit for the particular backing data type (unsigned short[], unsigned byte[])
     * @param image
     * @param width
     * @param height
     */
    protected AbstractImageData(int[] image, int width, int height)
    {
        if (image == null) {
            throw new IllegalArgumentException("image can not be null");
        }
        this.width = width;
        this.height = height;
    }
    
    /**
     * Returns whether the data values of the image are of byte[].
     * @return
     */
    public abstract boolean isCompressed();
    
    /**
     * Returns the data of the image as int[]. The values is of one of int's subtype, bounded by 
     * {@link getDataElementMaxValue()}.
     * @return
     */
    public int[] getData() {
        return this.image.getArray();
    }
    
    /**
     * Returns the maximal value that can be stored in the returned data array.
     * @return
     */
    public abstract int getDataElementMaxValue();
    
    /**
     * Sets the image data. Subclasses check the parameter whether it fits for 
     * the particular subtype (unsigned byte[], unsigned short[]).
     * @param data
     */
    public abstract void setData(int[] data);
    
    /**
     * Returns the image metadata.
     * @return
     */
    public AbstractMetadata getMetadata() {
        return this.metadata;
    }
    
    /**
     * Gets an InputStream of the backing array.
     * @return
     */
    public InputStream getInputStream() {
        return this.image.getInputStream();
    }
    
    /**
     * Returns width of the image.
     * @return
     */
    public int getWidth() {
        return this.width;
    }
    
    /**
     * Returns height of the image.
     * @return
     */
    public int getHeight() {
        return this.height;
    }
    
    //////////////////////////////////// GENERATED

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + height;
        result = prime * result + ((image == null) ? 0 : image.hashCode());
        result = prime * result + ((metadata == null) ? 0 : metadata.hashCode());
        result = prime * result + width;
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        AbstractImageData other = (AbstractImageData) obj;
        if (height != other.height)
            return false;
        if (image == null) {
            if (other.image != null)
                return false;
        } else if (!image.equals(other.image))
            return false;
        if (metadata == null) {
            if (other.metadata != null)
                return false;
        } else if (!metadata.equals(other.metadata))
            return false;
        if (width != other.width)
            return false;
        return true;
    }

    @Override
    public String toString() {
        return "AbstractImageData [width=" + width + ", height=" + height + ", image=" + image + ", metadata="
                + metadata + "]";
    }
    
}
