package de.pco.common;

/*-
 * #%L
 * pco-common
 * %%
 * Copyright (C) 2020 PCO
 * %%
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 * #L%
 */

import static org.junit.jupiter.api.Assertions.assertArrayEquals;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertThrows;

import org.junit.jupiter.api.Test;

/**
 * Tests the UShortArray and UByteArray
 * @author PCO
 *
 */
class UnsignedArrayTest 
{
    /**
     * Tests the constructor of UByteArray from the array of (signed) bytes.
     */
    @Test
    void testUByteArrayFromBytes() 
    {    
        byte[] byteArray = new byte[] {-1, -128, 127}; // we get bytes {255, 128, 127} in C
        UByteArray uByteArray = new UByteArray(byteArray);
        int[] intArray = uByteArray.getArray();
        assertArrayEquals(new int[]{255, 128, 127}, intArray);
        byte[] byteArray2 = uByteArray.getBackingArray();
        assertArrayEquals(byteArray,byteArray2);
        assertEquals(255, uByteArray.ELEMENT_MAX_VALUE);
    }

    /**
     * Tests the constructor of UByteArray from the array of real unsigned byte 
     * values stored in an int array.
     */
    @Test
    void testUByteArrayFromInts() 
    {    
        int[] intArray = new int[] {255, 128, 127}; // in C it would be {255, 128, 127}
        UByteArray uByteArray = new UByteArray(intArray);
        int[] intArray2 = uByteArray.getArray();
        assertArrayEquals(intArray, intArray2);
        byte[] byteArray = uByteArray.getBackingArray();
        assertArrayEquals(new byte[] {-1, -128, 127}, byteArray);
    }
    
    /**
     * Tests the range check by passing parameters outside <0, 255> 
     */
    @Test
    void testUByteArrayFromIntsErrors() 
    {
        int[] intArray = new int[] {0, 0, 256};
        assertThrows(IllegalArgumentException.class, () -> new UByteArray(intArray));
        int[] intArray1 = new int[] {-5, 0, 0};
        assertThrows(IllegalArgumentException.class, () -> new UByteArray(intArray1));
        int[] intArray2 = new int[] {-305, 5555, 68};
        assertThrows(IllegalArgumentException.class, () -> new UByteArray(intArray2));
    }
    
    /**
     * Tests the constructor of UShortArray from the array of (signed) shorts.
     */
    @Test
    void testUShortArrayFromShorts() 
    {    
        short[] shortArray = new short[] {-1, -32768, 127}; // we get bytes {65535, 32768, 127} in C
        UShortArray uShortArray = new UShortArray(shortArray);
        int[] intArray = uShortArray.getArray();
        assertArrayEquals(new int[]{65535, 32768, 127}, intArray);
        short[] shortArray2 = uShortArray.getBackingArray();
        assertArrayEquals(shortArray, shortArray2);
        assertEquals(65535, uShortArray.ELEMENT_MAX_VALUE);
    }

    /**
     * Tests the constructor of UShortArray from the array of real unsigned short values 
     * stored in an int array.
     */
    @Test
    void testUShortArrayFromInts() 
    {    
        int[] intArray = new int[] {65535, 32768, 127}; 
        UShortArray uShortArray = new UShortArray(intArray);
        int[] intArray2 = uShortArray.getArray();
        assertArrayEquals(intArray, intArray2);
        short[] shortArray = uShortArray.getBackingArray();
        assertArrayEquals(new short[] {-1, -32768, 127}, shortArray);
    }

    /**
     * Tests the range check by passing parameters outside <0, 65535> 
     */
    @Test
    void testUShortArrayFromIntsErrors() 
    {
        int[] intArray = new int[] {0, 0, 65536};
        assertThrows(IllegalArgumentException.class, () -> new UShortArray(intArray));
        int[] intArray1 = new int[] {-5, 0, 0};
        assertThrows(IllegalArgumentException.class, () -> new UShortArray(intArray1));
        int[] intArray2 = new int[] {66000, 5555, 68};
        assertThrows(IllegalArgumentException.class, () -> new UShortArray(intArray2));
    }
}
