/*
 * Copyright 2012 Stefan Haun, Andreas Nürnberger
 * 
 *      Data and Knowledge Engineering Group, 
 * 		Faculty of Computer Science,
 *		Otto-von-Guericke University,
 *		Magdeburg, Germany
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.ovgu.dke.mocca.util;

import java.util.Properties;

import net.jcip.annotations.ThreadSafe;

import de.ovgu.dke.mocca.api.MoccaException;
import de.ovgu.dke.mocca.api.command.Command;
import de.ovgu.dke.mocca.api.context.Context;
import de.ovgu.dke.mocca.api.context.State;

/**
 * An implementation of the {@link State} interface, that tracks changes and
 * creates the respective update command calls.
 * 
 * @author Stefan Haun (Stefan.haun@ovgu.de)
 * 
 */
@ThreadSafe
public class UpdatableState implements State {
	private final UpdatableProperty<String> text;
	private final UpdatableProperty<Lifecycle> lifecycle;
	private final UpdatableProperty<Integer> progress_min;
	private final UpdatableProperty<Integer> progress_max;
	private final UpdatableProperty<Integer> progress_current;
	private final UpdatableProperty<Severity> severity;
	private final UpdatableProperty<String> trace;

	/**
	 * Create a new instance with clean values set to null.
	 */
	public UpdatableState() {
		this.text = new UpdatableProperty<String>(PARA_TEXT);
		this.lifecycle = new UpdatableProperty<State.Lifecycle>(PARA_LIFECYCLE);
		this.progress_min = new UpdatableProperty<Integer>(PARA_MIN);
		this.progress_max = new UpdatableProperty<Integer>(PARA_MAX);
		this.progress_current = new UpdatableProperty<Integer>(PARA_CURRENT);
		this.severity = new UpdatableProperty<State.Severity>(PARA_SEVERITY);
		this.trace = new UpdatableProperty<String>(PARA_TRACE);
	}

	/**
	 * Get the text value.
	 * 
	 * The text is a short representation of the message's intent, often a
	 * keyword representing a state.
	 * 
	 * @return the value of the text property; null if none has been set.
	 */
	@Override
	public synchronized String getText() {
		return text.getValue();
	}

	/**
	 * Set the text value.
	 * 
	 * The text is a short representation of the message's intent, often a
	 * keyword representing a state.
	 * 
	 * @param text
	 *            the new value of the text property, null to delete it.
	 */
	@Override
	public synchronized void setText(String text) {
		this.text.setValue(text);
	}

	/**
	 * Get the life-cycle value.
	 * 
	 * The life-cycle represents the state of the context.
	 * 
	 * @return the value of the life-cycle property, one of {@link Lifecycle};
	 *         null if none has been set.
	 */
	@Override
	public synchronized Lifecycle getLifecycle() {
		return lifecycle.getValue();
	}

	/**
	 * Set the life-cycle value.
	 * 
	 * The life-cycle represents the state of the context.
	 * 
	 * @param lifecycle
	 *            the new value of the life-cycle property, null to delete it.
	 */
	@Override
	public synchronized void setLifecycle(Lifecycle lifecycle) {
		this.lifecycle.setValue(lifecycle);
	}

	/**
	 * Get the minimal progress value.
	 * 
	 * @return the value of the minimal-progress property, null if none has been
	 *         set.
	 */
	@Override
	public synchronized Integer getMinProgress() {
		return progress_min.getValue();
	}

	/**
	 * Set the minimal progress value.
	 * 
	 * @param progress_min
	 *            the minimal progress value, null to delete it.
	 */
	@Override
	public synchronized void setMinProgress(Integer progress_min) {
		this.progress_min.setValue(progress_min);
	}

	/**
	 * Get the maximal progress value.
	 * 
	 * @return the value of the maximal-progress property, null if none has been
	 *         set.
	 */
	@Override
	public synchronized Integer getMaxProgress() {
		return progress_max.getValue();
	}

	/**
	 * Set the maximal progress value.
	 * 
	 * @param progress_max
	 *            The new value of the maximal-progres property, null to delete
	 *            it.
	 */
	@Override
	public synchronized void setMaxProgress(Integer progress_max) {
		this.progress_max.setValue(progress_max);
	}

	/**
	 * Get the current progress value.
	 * 
	 * @return The value of the current-progress property, null if none has been
	 *         set.
	 */
	@Override
	public synchronized Integer getCurrentProgress() {
		return progress_current.getValue();
	}

	/**
	 * Set the current progress value.
	 * 
	 * @param progress_current
	 *            The new value of the current-progress property, null to delete
	 *            it.
	 */
	@Override
	public synchronized void setCurrentProgress(Integer progress_current) {
		this.progress_current.setValue(progress_current);
	}

	/**
	 * Get the severity of the state.
	 * 
	 * The severity can be used for log filtering and a generic handling of
	 * status updates.
	 * 
	 * @return The value of the severity property, null if none has been set.
	 */
	@Override
	public synchronized Severity getSeverity() {
		return severity.getValue();
	}

	/**
	 * Set the severity of the state.
	 * 
	 * The severity can be used for log filtering and a generic handling of
	 * status updates.
	 * 
	 * @param severity
	 *            The new value for the severity property, null to delete it.
	 */
	@Override
	public synchronized void setSeverity(Severity severity) {
		this.severity.setValue(severity);
	}

	/**
	 * Get the trace information of the state.
	 * 
	 * This property can be used to provide a more detailed description of the
	 * state, e.g. a Stack Trace from an {@link Exception}.
	 * 
	 * @return the value of the trace property, null if none has been set.
	 */
	@Override
	public synchronized String getTrace() {
		return trace.getValue();
	}

	/**
	 * Set the trace information of the state.
	 * 
	 * This property can be used to provide a more detailed description of the
	 * state, e.g. a Stack Trace from an {@link Exception}.
	 * 
	 * @param trace
	 *            The new value of the trace property, null to delete it.
	 */
	@Override
	public synchronized void setTrace(String trace) {
		this.trace.setValue(trace);
	}

	/**
	 * Check if a dirty flag has been set
	 * 
	 * @return true if some values have been changed
	 */
	protected synchronized boolean isDirty() {
		return this.lifecycle.isDirty() || this.progress_current.isDirty()
				|| this.progress_max.isDirty() || this.progress_min.isDirty()
				|| this.severity.isDirty() || this.text.isDirty()
				|| this.trace.isDirty();
	}

	/**
	 * reset all dirty flags
	 */
	protected synchronized void undirty() {
		this.text.undirty();
		this.lifecycle.undirty();
		this.progress_min.undirty();
		this.progress_max.undirty();
		this.progress_current.undirty();
		this.severity.undirty();
		this.trace.undirty();
	}

	/**
	 * Create an update-status command containing only the properties that have
	 * been changed since the last status update.
	 * 
	 * @param ctx
	 *            The context in which the command should be created.
	 * @return an update-status command with the changed status properties or
	 *         null, if the status is unchanged.
	 * @throws MoccaException
	 *             if command creation fails.
	 * @throws NullPointerException
	 *             if the context argument is null.
	 */
	@Override
	public synchronized Command createStateUpdate(Context ctx)
			throws MoccaException {
		if (!isDirty())
			return null;

		final Properties props = new Properties();
		// copy to properties if dirty flag is set
		this.text.putProperty(props);
		this.lifecycle.putProperty(props);
		this.progress_min.putProperty(props);
		this.progress_max.putProperty(props);
		this.progress_current.putProperty(props);
		this.severity.putProperty(props);
		this.trace.putProperty(props);

		// create command here to check for exceptions
		final Command cmd = ctx.getRuntime().createCommand(State.CMD_UPDATE,
				props);

		// now reset dirty flags
		undirty();

		return cmd;
	}
}
