/*
 * Copyright 2012 Stefan Haun, Andreas Nürnberger
 * 
 *      Data and Knowledge Engineering Group, 
 *      Faculty of Computer Science,
 *      Otto-von-Guericke University,
 *      Magdeburg, Germany
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.ovgu.dke.mocca.util;

import java.util.Properties;

import net.jcip.annotations.NotThreadSafe;

import de.ovgu.dke.mocca.api.context.State;

/**
 * A property that tracks if its value is changed.
 * 
 * @author Stefan Haun (stefan.haun@ovgu.de)
 * 
 * @param <T>
 *            Type of the property value
 */
@NotThreadSafe
public class UpdatableProperty<T> {
	private final String key;

	private T value;
	private boolean dirty = false;

	/**
	 * Create a new updatable property.
	 * 
	 * @param key
	 *            The key for the property.
	 * @throws NullPointerException
	 *             if the key argument is null.
	 */
	public UpdatableProperty(final String key) {
		this(key, null);
	}

	/**
	 * Create a new updatable property with a specific value.
	 * 
	 * @param key
	 *            The key for the property.
	 * @param value
	 *            The value, null if none provided so far.
	 * @throws NullPointerException
	 *             if the key argument is null.
	 */
	public UpdatableProperty(final String key, T value) {
		if (key == null)
			throw new NullPointerException("Key parameter may not be null!");

		this.key = key;
		this.value = value;
	}

	/**
	 * Get the property key.
	 * 
	 * @return The key value.
	 */
	public String getKey() {
		return key;
	}

	/**
	 * Get the property value.
	 * 
	 * @return The property value.
	 */
	public T getValue() {
		return value;
	}

	/**
	 * Set the property value.
	 * 
	 * @param value
	 *            the new value.
	 */
	public void setValue(T value) {
		dirty = !equalValues(this.value, value);
		this.value = value;
	}

	/**
	 * Check if the value has been updated since initialization or the last call
	 * to <code>undirty</code>.
	 * 
	 * @return true if the property value has been updated.
	 */
	public boolean isDirty() {
		return dirty;
	}

	/**
	 * Clear the dirty flag.
	 */
	public void undirty() {
		this.dirty = false;
	}

	/**
	 * Put a {@link String} represenation of the updatable property into a
	 * {@link Properties} instance, if the value has been changed.
	 * 
	 * The value will be either the result of a call to toString() on getValue()
	 * or, if getValue() returns null, the #nil value.
	 * 
	 * @param props
	 *            The properties instance into which the updatable property will
	 *            be stored.
	 * @return true, if the property has been stored, false otherwise.
	 */
	public boolean putProperty(final Properties props) {
		if (!isDirty())
			return false;

		props.put(getKey(), checkNull(getValue()));

		return true;
	}

	private final boolean equalValues(T v1, T v2) {
		return ((v1 == null) && (v2 == null))
				|| ((v1 != null) && v2.equals(v1))
				|| ((v2 != null) && v2.equals(v1));
	}

	private final String checkNull(final T val) {
		return val == null ? State.NIL : val.toString();
	}

	/**
	 * toString-value is taken from the value's toString-method
	 * 
	 * @return the toString() result from the value or "null", if the value is
	 *         set to null
	 */
	@Override
	public String toString() {
		return this.value == null ? "null" : this.value.toString();
	}
}
