/*
 * Copyright 2012 Stefan Haun, Andreas Nürnberger
 * 
 *      Data and Knowledge Engineering Group, 
 * 		Faculty of Computer Science,
 *		Otto-von-Guericke University,
 *		Magdeburg, Germany
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.ovgu.dke.mocca.util;

import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import de.ovgu.dke.mocca.api.MoccaException;
import de.ovgu.dke.mocca.api.MoccaRuntime;
import de.ovgu.dke.mocca.api.command.Command;
import de.ovgu.dke.mocca.glue.GlueMoccaRuntimeImpl;

/**
 * Support methods for the Mocca environment.
 * 
 * @author Stefan Haun (stefan.haun@ovgu.de)
 */
public class MoccaHelper {
	static Log logger = LogFactory.getLog(MoccaHelper.class);

	private static MoccaRuntime defaultRuntime = null;

	/**
	 * Instance control: instances are not allowed!
	 */
	private MoccaHelper() {
		super();
		// nothing to do
	}

	/**
	 * <p>
	 * Get the default runtime.
	 * </p>
	 * <p>
	 * When a context instance is available, you should obtain the runtime from
	 * the context!
	 * </p>
	 * 
	 * @return The default runtime instance.
	 */
	public static synchronized MoccaRuntime getDefaultRuntime() {
		if (defaultRuntime == null)
			defaultRuntime = new GlueMoccaRuntimeImpl();

		return defaultRuntime;
	}

	/**
	 * Add a shutdown hook to the default runtime. This allows to clean up mocca
	 * without the need of additional code in the program using Mocca.
	 * 
	 * @param mocca
	 *            The Mocca Runtime to be shut down.
	 * @throws NullPointerException
	 *             if the runtime argument is null.
	 */
	public static void addShutdownHook(final MoccaRuntime mocca) {
		if (mocca == null)
			throw new NullPointerException(
					"Mocca runtime argument must not be null!");

		Runtime.getRuntime().addShutdownHook(new Thread(new Runnable() {

			@Override
			public void run() {
				try {
					logger.info("Disposing mocca runtime.");
					mocca.dispose();
				} catch (MoccaException e) {
					logger.warn("Error shutting down mocca: " + e.getMessage(),
							e);
				}
			}
		}));
	}

	/**
	 * Create properties object from a commands parameter set
	 * 
	 * @param cmd
	 *            The command to convert.
	 * @return A @code{Properties} instance containing the parameters
	 * @throws NullPointerException
	 *             if the cmd parameter is @code{null}
	 */
	public static Properties cmdParams2Props(final Command cmd) {
		if (cmd == null)
			throw new NullPointerException("The cmd parameter may not be null!");

		final Properties props = new Properties();
		for (String key : cmd.getParameters())
			props.put(key, cmd.getParameter(key));

		return props;
	}

}
