/*
 * Copyright 2012 Stefan Haun, Andreas Nürnberger
 * 
 *      Data and Knowledge Engineering Group, 
 *      Faculty of Computer Science,
 *      Otto-von-Guericke University,
 *      Magdeburg, Germany
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.ovgu.dke.mocca.util;

import de.ovgu.dke.mocca.api.MoccaException;
import de.ovgu.dke.mocca.api.command.Command;
import de.ovgu.dke.mocca.api.context.Context;
import de.ovgu.dke.mocca.api.context.State;
import de.ovgu.dke.mocca.api.context.State.Lifecycle;
import de.ovgu.dke.mocca.api.context.State.Severity;

/**
 * Utility methods for context and command handlers
 * 
 * @author Stefan Haun (stefan.haun@ovgu.de)
 * 
 */
public class ContextUtils {
	/**
	 * Send a status update to the context's peer.
	 * 
	 * @param ctx
	 *            The context.
	 * @throws MoccaException
	 *             if sending fails.
	 * @throws NullPointerException
	 *             if the context argument is null.
	 */
	private static void sendStateUpdate(Context ctx) throws MoccaException {
		final Command cmd = ctx.getState().createStateUpdate(ctx);
		if (cmd != null)
			ctx.sendCommand(cmd);
	}

	/**
	 * Set a new life-cycle state and send update to the context's peer.
	 * 
	 * @param ctx
	 *            The context.
	 * @param lifecycle
	 *            The new status.
	 * @throws MoccaException
	 *             if sending fails.
	 * @throws NullPointerException
	 *             if the context argument is null.
	 */
	public static void sendProgress(Context ctx, Lifecycle lifecycle)
			throws MoccaException {
		final State state = ctx.getState();
		state.setText(null);
		state.setTrace(null);
		state.setLifecycle(lifecycle);
		sendStateUpdate(ctx);
	}

	/**
	 * Send a fatal error message to the context's peer and set the context to
	 * FAILED.
	 * 
	 * @param ctx
	 *            The context.
	 * @param cause
	 *            The cause if the error, may be null.
	 * @param msg
	 *            The error message, may be null.
	 * @throws MoccaException
	 *             if sending fails.
	 * @throws NullPointerException
	 *             if the context argument is null.
	 */
	public static void sendError(Context ctx, String cause, String msg)
			throws MoccaException {
		final State state = ctx.getState();
		state.setLifecycle(Lifecycle.FAILED);
		state.setSeverity(Severity.FATAL);
		state.setText(cause);
		state.setTrace(msg);
		sendStateUpdate(ctx);
	}

	/**
	 * <p>
	 * Retrieve an integer parameter from the command message.
	 * </p>
	 * 
	 * <p>
	 * If the parameter cannot be converted to an integer
	 * </p>
	 * <ul>
	 * <li>an error message is sent</li>
	 * <li>the resulting NumberFormatException is re-thrown</li>
	 * </ul>
	 * 
	 * @param ctx
	 *            Current MOCCA process.
	 * @param cmd
	 *            The command from which the parameter will be extracted.
	 * @param key
	 *            The parameter name
	 * @return an Integer containing the value or <code>null</code> if the
	 *         parameter does not exist.
	 * @throws NumberFormatException
	 *             in case of a conversion error.
	 * @throws MoccaException
	 *             if sending fails.
	 * @throws NullPointerException
	 *             if one of the arguments is null.
	 */
	public static Integer retrieveInteger(Context ctx, Command cmd, String key)
			throws NumberFormatException, MoccaException {
		try {
			final String _i = cmd.getParameter(key);
			return _i == null ? null : Integer.parseInt(_i);
		} catch (NumberFormatException nfe) {
			sendError(ctx, "IllegalArgument",
					"NumberFormatException: " + nfe.getMessage());
			throw nfe;
		}
	}
}
